import { Event, Transport } from '@sentry/types';
import { Dsn } from '@sentry/utils';
/**
 * Payload for a minidump request comprising a persistent file system path and
 * event metadata.
 */
export interface MinidumpRequest {
    /** Path to the minidump file. */
    path: string;
    /** Associated event data. */
    event: Event;
}
/**
 * A service that discovers Minidump crash reports and uploads them to Sentry.
 */
export declare class MinidumpUploader {
    private readonly _crashesDirectory;
    private readonly _cacheDirectory;
    private readonly _transport;
    /** The type of the Electron CrashReporter used to search for Minidumps. */
    private readonly _type;
    /** The sub-directory where crashpad dumps can be found */
    private readonly _crashpadSubDirectory;
    /** List of minidumps that have been found already. */
    private readonly _knownPaths;
    /**
     * Store to persist queued Minidumps beyond application crashes or lost
     * internet connection.
     */
    private readonly _queue;
    /** API object */
    private readonly _api;
    /**
     * Creates a new uploader instance.
     *
     * @param dsn The Sentry DSN.
     * @param crashesDirectory The directory Electron stores crashes in.
     * @param cacheDirectory A persistent directory to cache minidumps.
     */
    constructor(dsn: Dsn, _crashesDirectory: string, _cacheDirectory: string, _transport: Transport);
    /**
     * Returns the minidump endpoint in Sentry
     * @param dsn Dsn
     */
    static minidumpUrlFromDsn(dsn: Dsn): string;
    /**
     * Uploads a minidump file to Sentry.
     *
     * @param path Absolute path to the minidump file.
     * @param event Event data to attach to the minidump.
     * @returns A promise that resolves when the upload is complete.
     */
    uploadMinidump(request: MinidumpRequest): Promise<void>;
    /**
     * Searches for new, unknown minidump files in the crash directory.
     * @returns A promise that resolves to absolute paths of those dumps.
     */
    getNewMinidumps(): Promise<string[]>;
    /** Flushes locally cached minidumps from the queue. */
    flushQueue(): Promise<void>;
    /**
     * Helper to filter an array with asynchronous callbacks.
     *
     * @param array An array containing items to filter.
     * @param predicate An async predicate evaluated on every item.
     * @param thisArg Optional value passed as "this" into the callback.
     * @returns An array containing only values where the callback returned true.
     */
    private _filterAsync;
    /** Scans the Crashpad directory structure for minidump files. */
    private _scanCrashpadFolder;
    /** Attempts to remove the metadata file so Crashpad doesn't output `failed to stat report` errors to the console */
    private _deleteCrashpadMetadataFile;
    /** Scans the Breakpad directory structure for minidump files. */
    private _scanBreakpadFolder;
    /** Crudely parses the dump file from the Breakpad multipart file */
    private _parseBreakpadMultipartFile;
    /**
     * Enqueues a minidump with event information for later upload.
     * @param request The request containing a minidump and event info.
     */
    private _queueMinidump;
    /**
     * Create minidump request to dispatch to the transpoirt
     */
    private _toMinidumpRequest;
}
//# sourceMappingURL=uploader.d.ts.map