Object.defineProperty(exports, "__esModule", { value: true });
exports.Electron = void 0;
var tslib_1 = require("tslib");
var core_1 = require("@sentry/core");
var electron_1 = require("electron");
/** Electron integration that cleans up the event. */
var Electron = /** @class */ (function () {
    function Electron() {
        /**
         * @inheritDoc
         */
        this.name = Electron.id;
    }
    /**
     * @inheritDoc
     */
    Electron.prototype.setupOnce = function () {
        var _this = this;
        this._instrumentBreadcrumbs('app', electron_1.app, function (event) { return !event.startsWith('remote-'); });
        electron_1.app.once('ready', function () {
            // We can't access these until 'ready'
            _this._instrumentBreadcrumbs('Screen', electron_1.screen);
            _this._instrumentBreadcrumbs('PowerMonitor', electron_1.powerMonitor);
        });
        electron_1.app.on('web-contents-created', function (_, contents) {
            // SetImmediate is required for contents.id to be correct
            // https://github.com/electron/electron/issues/12036
            setImmediate(function () {
                if (contents.isDestroyed()) {
                    return;
                }
                var options = core_1.getCurrentHub().getClient().getOptions();
                var customName = options.getRendererName && options.getRendererName(contents);
                _this._instrumentBreadcrumbs(customName || "WebContents[" + contents.id + "]", contents, function (event) {
                    return ['dom-ready', 'load-url', 'destroyed'].includes(event);
                });
            });
        });
    };
    /**
     * Hooks into the Electron EventEmitter to capture breadcrumbs for the
     * specified events.
     */
    Electron.prototype._instrumentBreadcrumbs = function (category, emitter, shouldInclude) {
        var emit = emitter.emit.bind(emitter);
        emitter.emit = function (event) {
            var args = [];
            for (var _i = 1; _i < arguments.length; _i++) {
                args[_i - 1] = arguments[_i];
            }
            if (shouldInclude === undefined || shouldInclude(event)) {
                var breadcrumb = {
                    category: 'electron',
                    message: category + "." + event,
                    timestamp: new Date().getTime() / 1000,
                    type: 'ui',
                };
                var self_1 = core_1.getCurrentHub().getIntegration(Electron);
                if (self_1) {
                    core_1.getCurrentHub().addBreadcrumb(breadcrumb);
                }
            }
            return emit.apply(void 0, tslib_1.__spreadArray([event], tslib_1.__read(args)));
        };
    };
    /**
     * @inheritDoc
     */
    Electron.id = 'Electron';
    return Electron;
}());
exports.Electron = Electron;
//# sourceMappingURL=electron.js.map