﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/deadline/DeadlineRequest.h>
#include <aws/deadline/Deadline_EXPORTS.h>
#include <aws/deadline/model/UpdatedSessionActionInfo.h>

#include <utility>

namespace Aws {
namespace deadline {
namespace Model {

/**
 */
class UpdateWorkerScheduleRequest : public DeadlineRequest {
 public:
  AWS_DEADLINE_API UpdateWorkerScheduleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateWorkerSchedule"; }

  AWS_DEADLINE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The farm ID to update.</p>
   */
  inline const Aws::String& GetFarmId() const { return m_farmId; }
  inline bool FarmIdHasBeenSet() const { return m_farmIdHasBeenSet; }
  template <typename FarmIdT = Aws::String>
  void SetFarmId(FarmIdT&& value) {
    m_farmIdHasBeenSet = true;
    m_farmId = std::forward<FarmIdT>(value);
  }
  template <typename FarmIdT = Aws::String>
  UpdateWorkerScheduleRequest& WithFarmId(FarmIdT&& value) {
    SetFarmId(std::forward<FarmIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The fleet ID to update.</p>
   */
  inline const Aws::String& GetFleetId() const { return m_fleetId; }
  inline bool FleetIdHasBeenSet() const { return m_fleetIdHasBeenSet; }
  template <typename FleetIdT = Aws::String>
  void SetFleetId(FleetIdT&& value) {
    m_fleetIdHasBeenSet = true;
    m_fleetId = std::forward<FleetIdT>(value);
  }
  template <typename FleetIdT = Aws::String>
  UpdateWorkerScheduleRequest& WithFleetId(FleetIdT&& value) {
    SetFleetId(std::forward<FleetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The worker ID to update.</p>
   */
  inline const Aws::String& GetWorkerId() const { return m_workerId; }
  inline bool WorkerIdHasBeenSet() const { return m_workerIdHasBeenSet; }
  template <typename WorkerIdT = Aws::String>
  void SetWorkerId(WorkerIdT&& value) {
    m_workerIdHasBeenSet = true;
    m_workerId = std::forward<WorkerIdT>(value);
  }
  template <typename WorkerIdT = Aws::String>
  UpdateWorkerScheduleRequest& WithWorkerId(WorkerIdT&& value) {
    SetWorkerId(std::forward<WorkerIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The session actions associated with the worker schedule to update.</p>
   */
  inline const Aws::Map<Aws::String, UpdatedSessionActionInfo>& GetUpdatedSessionActions() const { return m_updatedSessionActions; }
  inline bool UpdatedSessionActionsHasBeenSet() const { return m_updatedSessionActionsHasBeenSet; }
  template <typename UpdatedSessionActionsT = Aws::Map<Aws::String, UpdatedSessionActionInfo>>
  void SetUpdatedSessionActions(UpdatedSessionActionsT&& value) {
    m_updatedSessionActionsHasBeenSet = true;
    m_updatedSessionActions = std::forward<UpdatedSessionActionsT>(value);
  }
  template <typename UpdatedSessionActionsT = Aws::Map<Aws::String, UpdatedSessionActionInfo>>
  UpdateWorkerScheduleRequest& WithUpdatedSessionActions(UpdatedSessionActionsT&& value) {
    SetUpdatedSessionActions(std::forward<UpdatedSessionActionsT>(value));
    return *this;
  }
  template <typename UpdatedSessionActionsKeyT = Aws::String, typename UpdatedSessionActionsValueT = UpdatedSessionActionInfo>
  UpdateWorkerScheduleRequest& AddUpdatedSessionActions(UpdatedSessionActionsKeyT&& key, UpdatedSessionActionsValueT&& value) {
    m_updatedSessionActionsHasBeenSet = true;
    m_updatedSessionActions.emplace(std::forward<UpdatedSessionActionsKeyT>(key), std::forward<UpdatedSessionActionsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_farmId;

  Aws::String m_fleetId;

  Aws::String m_workerId;

  Aws::Map<Aws::String, UpdatedSessionActionInfo> m_updatedSessionActions;
  bool m_farmIdHasBeenSet = false;
  bool m_fleetIdHasBeenSet = false;
  bool m_workerIdHasBeenSet = false;
  bool m_updatedSessionActionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace deadline
}  // namespace Aws
