﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/es/ElasticsearchServiceRequest.h>
#include <aws/es/ElasticsearchService_EXPORTS.h>
#include <aws/es/model/AdvancedSecurityOptionsInput.h>
#include <aws/es/model/AutoTuneOptionsInput.h>
#include <aws/es/model/CognitoOptions.h>
#include <aws/es/model/DomainEndpointOptions.h>
#include <aws/es/model/EBSOptions.h>
#include <aws/es/model/ElasticsearchClusterConfig.h>
#include <aws/es/model/EncryptionAtRestOptions.h>
#include <aws/es/model/LogPublishingOption.h>
#include <aws/es/model/LogType.h>
#include <aws/es/model/NodeToNodeEncryptionOptions.h>
#include <aws/es/model/SnapshotOptions.h>
#include <aws/es/model/Tag.h>
#include <aws/es/model/VPCOptions.h>

#include <utility>

namespace Aws {
namespace ElasticsearchService {
namespace Model {

/**
 */
class CreateElasticsearchDomainRequest : public ElasticsearchServiceRequest {
 public:
  AWS_ELASTICSEARCHSERVICE_API CreateElasticsearchDomainRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateElasticsearchDomain"; }

  AWS_ELASTICSEARCHSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the Elasticsearch domain that you are creating. Domain names are
   * unique across the domains owned by an account within an AWS region. Domain names
   * must start with a lowercase letter and can contain the following characters: a-z
   * (lowercase), 0-9, and - (hyphen).</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  CreateElasticsearchDomainRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>String of format X.Y to specify version for the Elasticsearch domain eg.
   * "1.5" or "2.3". For more information, see <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomains"
   * target="_blank">Creating Elasticsearch Domains</a> in the <i>Amazon
   * Elasticsearch Service Developer Guide</i>.</p>
   */
  inline const Aws::String& GetElasticsearchVersion() const { return m_elasticsearchVersion; }
  inline bool ElasticsearchVersionHasBeenSet() const { return m_elasticsearchVersionHasBeenSet; }
  template <typename ElasticsearchVersionT = Aws::String>
  void SetElasticsearchVersion(ElasticsearchVersionT&& value) {
    m_elasticsearchVersionHasBeenSet = true;
    m_elasticsearchVersion = std::forward<ElasticsearchVersionT>(value);
  }
  template <typename ElasticsearchVersionT = Aws::String>
  CreateElasticsearchDomainRequest& WithElasticsearchVersion(ElasticsearchVersionT&& value) {
    SetElasticsearchVersion(std::forward<ElasticsearchVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration options for an Elasticsearch domain. Specifies the instance
   * type and number of instances in the domain cluster. </p>
   */
  inline const ElasticsearchClusterConfig& GetElasticsearchClusterConfig() const { return m_elasticsearchClusterConfig; }
  inline bool ElasticsearchClusterConfigHasBeenSet() const { return m_elasticsearchClusterConfigHasBeenSet; }
  template <typename ElasticsearchClusterConfigT = ElasticsearchClusterConfig>
  void SetElasticsearchClusterConfig(ElasticsearchClusterConfigT&& value) {
    m_elasticsearchClusterConfigHasBeenSet = true;
    m_elasticsearchClusterConfig = std::forward<ElasticsearchClusterConfigT>(value);
  }
  template <typename ElasticsearchClusterConfigT = ElasticsearchClusterConfig>
  CreateElasticsearchDomainRequest& WithElasticsearchClusterConfig(ElasticsearchClusterConfigT&& value) {
    SetElasticsearchClusterConfig(std::forward<ElasticsearchClusterConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to enable, disable and specify the type and size of EBS storage
   * volumes. </p>
   */
  inline const EBSOptions& GetEBSOptions() const { return m_eBSOptions; }
  inline bool EBSOptionsHasBeenSet() const { return m_eBSOptionsHasBeenSet; }
  template <typename EBSOptionsT = EBSOptions>
  void SetEBSOptions(EBSOptionsT&& value) {
    m_eBSOptionsHasBeenSet = true;
    m_eBSOptions = std::forward<EBSOptionsT>(value);
  }
  template <typename EBSOptionsT = EBSOptions>
  CreateElasticsearchDomainRequest& WithEBSOptions(EBSOptionsT&& value) {
    SetEBSOptions(std::forward<EBSOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> IAM access policy as a JSON-formatted string.</p>
   */
  inline const Aws::String& GetAccessPolicies() const { return m_accessPolicies; }
  inline bool AccessPoliciesHasBeenSet() const { return m_accessPoliciesHasBeenSet; }
  template <typename AccessPoliciesT = Aws::String>
  void SetAccessPolicies(AccessPoliciesT&& value) {
    m_accessPoliciesHasBeenSet = true;
    m_accessPolicies = std::forward<AccessPoliciesT>(value);
  }
  template <typename AccessPoliciesT = Aws::String>
  CreateElasticsearchDomainRequest& WithAccessPolicies(AccessPoliciesT&& value) {
    SetAccessPolicies(std::forward<AccessPoliciesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Option to set time, in UTC format, of the daily automated snapshot. Default
   * value is 0 hours. </p>
   */
  inline const SnapshotOptions& GetSnapshotOptions() const { return m_snapshotOptions; }
  inline bool SnapshotOptionsHasBeenSet() const { return m_snapshotOptionsHasBeenSet; }
  template <typename SnapshotOptionsT = SnapshotOptions>
  void SetSnapshotOptions(SnapshotOptionsT&& value) {
    m_snapshotOptionsHasBeenSet = true;
    m_snapshotOptions = std::forward<SnapshotOptionsT>(value);
  }
  template <typename SnapshotOptionsT = SnapshotOptions>
  CreateElasticsearchDomainRequest& WithSnapshotOptions(SnapshotOptionsT&& value) {
    SetSnapshotOptions(std::forward<SnapshotOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify the subnets and security groups for VPC endpoint. For more
   * information, see <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-creating-vpc"
   * target="_blank">Creating a VPC</a> in <i>VPC Endpoints for Amazon Elasticsearch
   * Service Domains</i></p>
   */
  inline const VPCOptions& GetVPCOptions() const { return m_vPCOptions; }
  inline bool VPCOptionsHasBeenSet() const { return m_vPCOptionsHasBeenSet; }
  template <typename VPCOptionsT = VPCOptions>
  void SetVPCOptions(VPCOptionsT&& value) {
    m_vPCOptionsHasBeenSet = true;
    m_vPCOptions = std::forward<VPCOptionsT>(value);
  }
  template <typename VPCOptionsT = VPCOptions>
  CreateElasticsearchDomainRequest& WithVPCOptions(VPCOptionsT&& value) {
    SetVPCOptions(std::forward<VPCOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify the Cognito user and identity pools for Kibana
   * authentication. For more information, see <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
   * target="_blank">Amazon Cognito Authentication for Kibana</a>.</p>
   */
  inline const CognitoOptions& GetCognitoOptions() const { return m_cognitoOptions; }
  inline bool CognitoOptionsHasBeenSet() const { return m_cognitoOptionsHasBeenSet; }
  template <typename CognitoOptionsT = CognitoOptions>
  void SetCognitoOptions(CognitoOptionsT&& value) {
    m_cognitoOptionsHasBeenSet = true;
    m_cognitoOptions = std::forward<CognitoOptionsT>(value);
  }
  template <typename CognitoOptionsT = CognitoOptions>
  CreateElasticsearchDomainRequest& WithCognitoOptions(CognitoOptionsT&& value) {
    SetCognitoOptions(std::forward<CognitoOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Encryption At Rest Options.</p>
   */
  inline const EncryptionAtRestOptions& GetEncryptionAtRestOptions() const { return m_encryptionAtRestOptions; }
  inline bool EncryptionAtRestOptionsHasBeenSet() const { return m_encryptionAtRestOptionsHasBeenSet; }
  template <typename EncryptionAtRestOptionsT = EncryptionAtRestOptions>
  void SetEncryptionAtRestOptions(EncryptionAtRestOptionsT&& value) {
    m_encryptionAtRestOptionsHasBeenSet = true;
    m_encryptionAtRestOptions = std::forward<EncryptionAtRestOptionsT>(value);
  }
  template <typename EncryptionAtRestOptionsT = EncryptionAtRestOptions>
  CreateElasticsearchDomainRequest& WithEncryptionAtRestOptions(EncryptionAtRestOptionsT&& value) {
    SetEncryptionAtRestOptions(std::forward<EncryptionAtRestOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the NodeToNodeEncryptionOptions.</p>
   */
  inline const NodeToNodeEncryptionOptions& GetNodeToNodeEncryptionOptions() const { return m_nodeToNodeEncryptionOptions; }
  inline bool NodeToNodeEncryptionOptionsHasBeenSet() const { return m_nodeToNodeEncryptionOptionsHasBeenSet; }
  template <typename NodeToNodeEncryptionOptionsT = NodeToNodeEncryptionOptions>
  void SetNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptionsT&& value) {
    m_nodeToNodeEncryptionOptionsHasBeenSet = true;
    m_nodeToNodeEncryptionOptions = std::forward<NodeToNodeEncryptionOptionsT>(value);
  }
  template <typename NodeToNodeEncryptionOptionsT = NodeToNodeEncryptionOptions>
  CreateElasticsearchDomainRequest& WithNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptionsT&& value) {
    SetNodeToNodeEncryptionOptions(std::forward<NodeToNodeEncryptionOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Option to allow references to indices in an HTTP request body. Must be
   * <code>false</code> when configuring access to individual sub-resources. By
   * default, the value is <code>true</code>. See <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-advanced-options"
   * target="_blank">Configuration Advanced Options</a> for more information.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdvancedOptions() const { return m_advancedOptions; }
  inline bool AdvancedOptionsHasBeenSet() const { return m_advancedOptionsHasBeenSet; }
  template <typename AdvancedOptionsT = Aws::Map<Aws::String, Aws::String>>
  void SetAdvancedOptions(AdvancedOptionsT&& value) {
    m_advancedOptionsHasBeenSet = true;
    m_advancedOptions = std::forward<AdvancedOptionsT>(value);
  }
  template <typename AdvancedOptionsT = Aws::Map<Aws::String, Aws::String>>
  CreateElasticsearchDomainRequest& WithAdvancedOptions(AdvancedOptionsT&& value) {
    SetAdvancedOptions(std::forward<AdvancedOptionsT>(value));
    return *this;
  }
  template <typename AdvancedOptionsKeyT = Aws::String, typename AdvancedOptionsValueT = Aws::String>
  CreateElasticsearchDomainRequest& AddAdvancedOptions(AdvancedOptionsKeyT&& key, AdvancedOptionsValueT&& value) {
    m_advancedOptionsHasBeenSet = true;
    m_advancedOptions.emplace(std::forward<AdvancedOptionsKeyT>(key), std::forward<AdvancedOptionsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Map of <code>LogType</code> and <code>LogPublishingOption</code>, each
   * containing options to publish a given type of Elasticsearch log.</p>
   */
  inline const Aws::Map<LogType, LogPublishingOption>& GetLogPublishingOptions() const { return m_logPublishingOptions; }
  inline bool LogPublishingOptionsHasBeenSet() const { return m_logPublishingOptionsHasBeenSet; }
  template <typename LogPublishingOptionsT = Aws::Map<LogType, LogPublishingOption>>
  void SetLogPublishingOptions(LogPublishingOptionsT&& value) {
    m_logPublishingOptionsHasBeenSet = true;
    m_logPublishingOptions = std::forward<LogPublishingOptionsT>(value);
  }
  template <typename LogPublishingOptionsT = Aws::Map<LogType, LogPublishingOption>>
  CreateElasticsearchDomainRequest& WithLogPublishingOptions(LogPublishingOptionsT&& value) {
    SetLogPublishingOptions(std::forward<LogPublishingOptionsT>(value));
    return *this;
  }
  inline CreateElasticsearchDomainRequest& AddLogPublishingOptions(LogType key, LogPublishingOption value) {
    m_logPublishingOptionsHasBeenSet = true;
    m_logPublishingOptions.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify configuration that will be applied to the domain
   * endpoint.</p>
   */
  inline const DomainEndpointOptions& GetDomainEndpointOptions() const { return m_domainEndpointOptions; }
  inline bool DomainEndpointOptionsHasBeenSet() const { return m_domainEndpointOptionsHasBeenSet; }
  template <typename DomainEndpointOptionsT = DomainEndpointOptions>
  void SetDomainEndpointOptions(DomainEndpointOptionsT&& value) {
    m_domainEndpointOptionsHasBeenSet = true;
    m_domainEndpointOptions = std::forward<DomainEndpointOptionsT>(value);
  }
  template <typename DomainEndpointOptionsT = DomainEndpointOptions>
  CreateElasticsearchDomainRequest& WithDomainEndpointOptions(DomainEndpointOptionsT&& value) {
    SetDomainEndpointOptions(std::forward<DomainEndpointOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies advanced security options.</p>
   */
  inline const AdvancedSecurityOptionsInput& GetAdvancedSecurityOptions() const { return m_advancedSecurityOptions; }
  inline bool AdvancedSecurityOptionsHasBeenSet() const { return m_advancedSecurityOptionsHasBeenSet; }
  template <typename AdvancedSecurityOptionsT = AdvancedSecurityOptionsInput>
  void SetAdvancedSecurityOptions(AdvancedSecurityOptionsT&& value) {
    m_advancedSecurityOptionsHasBeenSet = true;
    m_advancedSecurityOptions = std::forward<AdvancedSecurityOptionsT>(value);
  }
  template <typename AdvancedSecurityOptionsT = AdvancedSecurityOptionsInput>
  CreateElasticsearchDomainRequest& WithAdvancedSecurityOptions(AdvancedSecurityOptionsT&& value) {
    SetAdvancedSecurityOptions(std::forward<AdvancedSecurityOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies Auto-Tune options.</p>
   */
  inline const AutoTuneOptionsInput& GetAutoTuneOptions() const { return m_autoTuneOptions; }
  inline bool AutoTuneOptionsHasBeenSet() const { return m_autoTuneOptionsHasBeenSet; }
  template <typename AutoTuneOptionsT = AutoTuneOptionsInput>
  void SetAutoTuneOptions(AutoTuneOptionsT&& value) {
    m_autoTuneOptionsHasBeenSet = true;
    m_autoTuneOptions = std::forward<AutoTuneOptionsT>(value);
  }
  template <typename AutoTuneOptionsT = AutoTuneOptionsInput>
  CreateElasticsearchDomainRequest& WithAutoTuneOptions(AutoTuneOptionsT&& value) {
    SetAutoTuneOptions(std::forward<AutoTuneOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <code>Tag</code> added during domain creation.</p>
   */
  inline const Aws::Vector<Tag>& GetTagList() const { return m_tagList; }
  inline bool TagListHasBeenSet() const { return m_tagListHasBeenSet; }
  template <typename TagListT = Aws::Vector<Tag>>
  void SetTagList(TagListT&& value) {
    m_tagListHasBeenSet = true;
    m_tagList = std::forward<TagListT>(value);
  }
  template <typename TagListT = Aws::Vector<Tag>>
  CreateElasticsearchDomainRequest& WithTagList(TagListT&& value) {
    SetTagList(std::forward<TagListT>(value));
    return *this;
  }
  template <typename TagListT = Tag>
  CreateElasticsearchDomainRequest& AddTagList(TagListT&& value) {
    m_tagListHasBeenSet = true;
    m_tagList.emplace_back(std::forward<TagListT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;

  Aws::String m_elasticsearchVersion;

  ElasticsearchClusterConfig m_elasticsearchClusterConfig;

  EBSOptions m_eBSOptions;

  Aws::String m_accessPolicies;

  SnapshotOptions m_snapshotOptions;

  VPCOptions m_vPCOptions;

  CognitoOptions m_cognitoOptions;

  EncryptionAtRestOptions m_encryptionAtRestOptions;

  NodeToNodeEncryptionOptions m_nodeToNodeEncryptionOptions;

  Aws::Map<Aws::String, Aws::String> m_advancedOptions;

  Aws::Map<LogType, LogPublishingOption> m_logPublishingOptions;

  DomainEndpointOptions m_domainEndpointOptions;

  AdvancedSecurityOptionsInput m_advancedSecurityOptions;

  AutoTuneOptionsInput m_autoTuneOptions;

  Aws::Vector<Tag> m_tagList;
  bool m_domainNameHasBeenSet = false;
  bool m_elasticsearchVersionHasBeenSet = false;
  bool m_elasticsearchClusterConfigHasBeenSet = false;
  bool m_eBSOptionsHasBeenSet = false;
  bool m_accessPoliciesHasBeenSet = false;
  bool m_snapshotOptionsHasBeenSet = false;
  bool m_vPCOptionsHasBeenSet = false;
  bool m_cognitoOptionsHasBeenSet = false;
  bool m_encryptionAtRestOptionsHasBeenSet = false;
  bool m_nodeToNodeEncryptionOptionsHasBeenSet = false;
  bool m_advancedOptionsHasBeenSet = false;
  bool m_logPublishingOptionsHasBeenSet = false;
  bool m_domainEndpointOptionsHasBeenSet = false;
  bool m_advancedSecurityOptionsHasBeenSet = false;
  bool m_autoTuneOptionsHasBeenSet = false;
  bool m_tagListHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticsearchService
}  // namespace Aws
