// license:BSD-3-Clause
// copyright-holders:Curt Coder
/***************************************************************************

    Motorola MC2661/MC68661 Enhanced Programmable Communications Interface

    Copyright the MESS Team.
    Visit http://mamedev.org for licensing and usage restrictions.

****************************************************************************
                            _____   _____
                    D2   1 |*    \_/     | 28  D1
                    D3   2 |             | 27  D0
                   RxD   3 |             | 26  Vcc
                   GND   4 |             | 25  _RxC/BKDET
                    D4   5 |             | 24  _DTR
                    D5   6 |             | 23  _RTS
                    D6   7 |   MC2661    | 22  _DSR
                    D7   8 |   MC68661   | 21  RESET
            _TxC/XSYNC   9 |             | 20  BRCLK
                    A1  10 |             | 19  TxD
                   _CE  11 |             | 18  _TxEMT/DSCHG
                    A0  12 |             | 17  _CTS
                  _R/W  13 |             | 16  _DCD
                _RxRDY  14 |_____________| 15  _TxRDY

***************************************************************************/

#pragma once

#ifndef __MC2661__
#define __MC2661__

#include "emu.h"



/***************************************************************************
    DEVICE CONFIGURATION MACROS
***************************************************************************/

#define MCFG_MC2661_ADD(_tag, _clock, _config) \
	MCFG_DEVICE_ADD(_tag, MC2661, _clock) \
	MCFG_DEVICE_CONFIG(_config)


#define MC2661_INTERFACE(_name) \
	const mc2661_interface (_name) =



/***************************************************************************
    TYPE DEFINITIONS
***************************************************************************/

// ======================> mc2661_interface

struct mc2661_interface
{
	int m_rxc;
	int m_txc;

	devcb_read_line     m_in_rxd_cb;
	devcb_write_line    m_out_txd_cb;

	devcb_write_line    m_out_rxrdy_cb;
	devcb_write_line    m_out_txrdy_cb;
	devcb_write_line    m_out_rts_cb;
	devcb_write_line    m_out_dtr_cb;
	devcb_write_line    m_out_txemt_dschg_cb;
	devcb_write_line    m_out_bkdet_cb;
	devcb_write_line    m_out_xsync_cb;
};


// ======================> mc2661_device

class mc2661_device :  public device_t,
						public device_serial_interface,
						public mc2661_interface
{
public:
	// construction/destruction
	mc2661_device(const machine_config &mconfig, const char *tag, device_t *owner, UINT32 clock);

	DECLARE_READ8_MEMBER( read );
	DECLARE_WRITE8_MEMBER( write );

	DECLARE_WRITE_LINE_MEMBER( rxc_w );
	DECLARE_WRITE_LINE_MEMBER( txc_w );

	DECLARE_WRITE_LINE_MEMBER( dsr_w );
	DECLARE_WRITE_LINE_MEMBER( dcd_w );
	DECLARE_WRITE_LINE_MEMBER( cts_w );

	DECLARE_READ_LINE_MEMBER( rxrdy_r );
	DECLARE_READ_LINE_MEMBER( txemt_r );

protected:
	// device-level overrides
	virtual void device_config_complete();
	virtual void device_start();
	virtual void device_reset();

	// device_serial_interface overrides
	virtual void tra_callback();
	virtual void tra_complete();
	virtual void rcv_callback();
	virtual void rcv_complete();
	virtual void input_callback(UINT8 state);

private:
	devcb_resolved_read_line    m_in_rxd_func;
	devcb_resolved_write_line   m_out_txd_func;

	devcb_resolved_write_line   m_out_rxrdy_func;
	devcb_resolved_write_line   m_out_txrdy_func;
	devcb_resolved_write_line   m_out_rts_func;
	devcb_resolved_write_line   m_out_dtr_func;
	devcb_resolved_write_line   m_out_txemt_dschg_func;
	devcb_resolved_write_line   m_out_bkdet_func;
	devcb_resolved_write_line   m_out_xsync_func;

	UINT8 m_rhr;
	UINT8 m_thr;
	UINT8 m_cr;
	UINT8 m_sr;
	UINT8 m_mr[2];
	UINT8 m_sync[3];

	int m_mode_index;
	int m_sync_index;
};


// device type definition
extern const device_type MC2661;



#endif
