//
// Syd: rock-solid application kernel
// src/kernel/net/bind.rs: bind(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    net::IpAddr,
    os::fd::{AsRawFd, OwnedFd},
};

use ipnet::IpNet;
use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::{
        socket::{getsockname, AddressFamily, SockaddrLike, SockaddrStorage},
        stat::umask,
    },
    unistd::fchdir,
};

use crate::{
    cookie::safe_bind,
    fs::CanonicalPath,
    hook::UNotifyEventRequest,
    path::XPathBuf,
    proc::proc_umask,
    sandbox::{Action, AddressPattern, Capability, CidrRule},
    wildmatch::MatchMethod,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn handle_bind(
    fd: OwnedFd,
    addr: &SockaddrStorage,
    root: Option<&CanonicalPath>,
    request: &UNotifyEventRequest,
    allow_safe_bind: bool,
) -> Result<ScmpNotifResp, Errno> {
    if addr.as_unix_addr().and_then(|a| a.path()).is_some() {
        let req = request.scmpreq;
        let mask = proc_umask(req.pid())?;

        // SAFETY:
        // 1. Honour directory for too long sockets.
        //    Note, the current working directory is per-thread here.
        // 2. We cannot resolve symlinks in root or we risk TOCTOU!
        #[allow(clippy::disallowed_methods)]
        let dirfd = root.as_ref().unwrap().dir.as_ref().unwrap();
        fchdir(dirfd)?;

        // SAFETY: Honour process' umask.
        // Note, the umask is per-thread here.
        umask(mask);

        // SAFETY: bind() does not work through dangling
        // symbolic links even with SO_REUSEADDR. When called
        // with a dangling symlink as argument, bind() fails
        // with EADDRINUSE unlike creat() which is going to
        // attempt to create the symlink target. Hence basename
        // in addr here is not vulnerable to TOCTOU.
        safe_bind(&fd, addr)?;
    } else {
        // SAFETY: addr is not a UNIX domain socket.
        safe_bind(&fd, addr)?;
    }

    // Handle allow_safe_bind and bind_map.
    // Ignore errors as bind has already succeeded.
    //
    // Configure sandbox:
    // Remove and re-add the address so repeated binds to the
    // same address cannot overflow the vector.
    let _result = (|fd: OwnedFd, request: &UNotifyEventRequest| -> Result<(), Errno> {
        let (addr, port) = match addr.family() {
            Some(AddressFamily::Unix) => {
                let addr = addr.as_unix_addr().ok_or(Errno::EINVAL)?;
                let unix = match (addr.path(), addr.as_abstract()) {
                    (Some(_), _) => {
                        // Case 1: UNIX domain socket

                        // SAFETY: addr.path()=Some asserts root is Some.
                        #[allow(clippy::disallowed_methods)]
                        let unix = root.unwrap().abs();

                        // Handle bind_map after successful bind for UNIX sockets.
                        // We ignore errors because there's nothing we can do
                        // about them.
                        let _ = request.add_bind(&fd, unix);
                        drop(fd); // Close our copy of the socket.

                        if !allow_safe_bind {
                            return Ok(());
                        }

                        unix.to_owned()
                    }
                    (_, Some(path)) => {
                        // Case 2: UNIX abstract socket

                        drop(fd); // Close our copy of the socket.

                        if !allow_safe_bind {
                            return Ok(());
                        }

                        // SAFETY: Prefix UNIX abstract sockets with `@' before access check.
                        let mut unix = XPathBuf::from("@");
                        let null = memchr::memchr(0, path).unwrap_or(path.len());
                        unix.append_bytes(&path[..null]);

                        unix
                    }
                    _ => {
                        // Case 3: unnamed UNIX socket.
                        //
                        // SAFETY: Use dummy path `!unnamed' for unnamed UNIX sockets.
                        XPathBuf::from("!unnamed")
                    }
                };

                let mut sandbox = request.get_mut_sandbox();
                let acl = sandbox.get_acl_mut(Capability::CAP_NET_CONNECT);
                if let Some(idx) = acl.iter().position(|(p, m, a)| {
                    *m == MatchMethod::Literal && *a == Action::Allow && p.is_equal(unix.as_bytes())
                }) {
                    acl.remove(idx);
                }
                return acl.push_front((unix, MatchMethod::Literal, Action::Allow));
            }
            Some(AddressFamily::Inet) => {
                if !allow_safe_bind {
                    return Ok(());
                }

                let addr = addr.as_sockaddr_in().ok_or(Errno::EINVAL)?;
                let mut port = addr.port();

                let addr = IpNet::new_assert(IpAddr::V4(addr.ip()), 32);
                if port == 0 {
                    port = getsockname::<SockaddrStorage>(fd.as_raw_fd())?
                        .as_sockaddr_in()
                        .ok_or(Errno::EINVAL)?
                        .port();
                }
                drop(fd); // Close our copy of the socket.

                (addr, port)
            }
            Some(AddressFamily::Inet6) => {
                if !allow_safe_bind {
                    return Ok(());
                }

                let addr = addr.as_sockaddr_in6().ok_or(Errno::EINVAL)?;
                let mut port = addr.port();

                let addr = IpNet::new_assert(IpAddr::V6(addr.ip()), 128);
                if port == 0 {
                    port = getsockname::<SockaddrStorage>(fd.as_raw_fd())?
                        .as_sockaddr_in6()
                        .ok_or(Errno::EINVAL)?
                        .port();
                }
                drop(fd); // Close our copy of the socket.

                (addr, port)
            }
            _ => return Ok(()),
        };

        let rule = CidrRule {
            act: Action::Allow,
            cap: Capability::CAP_NET_CONNECT,
            pat: AddressPattern {
                addr,
                port: Some(port..=port),
            },
        };

        let mut sandbox = request.get_mut_sandbox();
        if let Some(idx) = sandbox.cidr_rules.iter().position(|r| *r == rule) {
            sandbox.cidr_rules.remove(idx);
        }
        sandbox.cidr_rules.push_front(rule)?;

        // 1. The sandbox lock will be released on drop here.
        // 2. The socket fd will be closed on drop here.
        Ok(())
    })(fd, request);

    Ok(request.return_syscall(0))
}
