use strict;
use warnings;

package Footprintless::Util;
$Footprintless::Util::VERSION = '1.01';
# ABSTRACT: A utility method package for common functionality in Footprintless
# PODNAME: Footprintless::Util

use Carp;
use Exporter qw(import);

our @EXPORT_OK = qw(
    agent
    default_command_runner
    dumper
    exit_due_to
    slurp
    spurt
);

sub agent {
    require LWP::UserAgent;
    my $agent = LWP::UserAgent->new();
    $agent->env_proxy();
    return $agent;
}

sub default_command_runner {
    require Footprintless::CommandRunner::IPCRun;
    return Footprintless::CommandRunner::IPCRun->new(@_);
}

sub dumper {
    require Data::Dumper;
    my $dumper = Data::Dumper->new( \@_ );
    $dumper->Indent(1);
    return $dumper->Dump(@_);
}

sub exit_due_to {
    my ($dollar_at) = @_;
    if ( ref($dollar_at)
        && $dollar_at->isa('Footprintless::CommandRunner::Exception') )
    {
        $@->exit();
    }
    else {
        print( STDERR "$@" );
        exit 255;
    }
}

sub slurp {
    my ($file) = @_;

    # http://www.perl.com/pub/2003/11/21/slurp.html
    return do { local ( @ARGV, $/ ) = $file; <> };
}

sub spurt {
    my ( $content, $file, %options ) = @_;
    my $write_mode = $options{append} ? '>>' : '>';
    open( my $handle, $write_mode, $file )
        || croak("unable to open [$file]: $!");
    print( $handle $content );
    close($handle);
}

1;

__END__

=pod

=head1 NAME

Footprintless::Util - A utility method package for common functionality in Footprintless

=head1 VERSION

version 1.01

=head1 SYNOPSIS

  use Footprintless::Util qw(
      agent
      default_command_runner
      dumper
      exit_due_to
      slurp
      spurt
  );

  my $agent = agent();
  my $command_runner = default_command_runner();
  my $dumper = dumper();
  
  eval {
      $command_runner->run_or_die('cat /foo/bar');
  };
  exit_due_to($@) if ($@);

  my $content = slurp('/foo/bar');
  spurt('baz', '/foo/bar', append => 1);

=head1 DESCRIPTION

This module contains common utility methods used by Footprintless.

=head1 FUNCTIONS

=head2 agent()

Returns a new instance of C<LWP::UserAgent> configured with C<env_proxy>.

=head2 default_command_runner()

Returns a new instance of the default implementation of 
C<Footprintless::CommandRunner>.

=head2 dumper(@to_dump)

Prints a dump of C<@to_dump> using C<Data::Dumper> with C<Data::Dumper::Indent>
set to 1.

=head2 exit_due_to($reason)

If C<$reason> is an instance of C<Footprintless::CommandRunner::Exception>,
C<$reason-E<gt>exit()> will be called.  Otherwise, C<$reason> will be printed
to C<STDERR> and C<exit(255)> will be called.

=head2 slurp($file)

Reads the entire contents of C<$file> in one gulp.

=head2 spurt($content, $file, %options)

Writes C<$content> to C<$file>.  The available options are:

=over 4

=item append

If I<truthy>, C<$content> will be appended to C<$file> instead of overwriting.

=back

=head1 AUTHOR

Lucas Theisen <lucastheisen@pastdev.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Lucas Theisen.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<Footprintless|Footprintless>

=back

=cut
