flipMatrix <- function(x) x[, rev(seq_len(NROW(x)))]

averageElongationVectors <- function(x, nbins=30) {
    x.bin <- y.bin <- seq(0, 512, length=nbins)
    Ex <- Ey <- matrix(NA, nrow=nbins, ncol=nbins)
    intX <- findInterval(x$x.0.m.cx, x.bin)
    intY <- findInterval(x$x.0.m.cy, y.bin)
    for (i in seq_len(nbins))
        for (j in seq_len(nbins)) {
            inInterval <- intX == i & intY == j
            if (any(inInterval)) {
                Ex[i, j] <- mean(x$e.x[inInterval])
                Ey[i, j] <- mean(x$e.y[inInterval])
            }
        }
    return(list(Ex=flipMatrix(Ex), Ey=flipMatrix(Ey)))
}

averageFeature <- function(x, nbins=20, feature="e.x", w=512) {
    x <- as.tbl(x)
    x.bin <- y.bin <- seq(0, w, length=nbins)
    M <- matrix(NA, nrow=nbins, ncol=nbins)
    intX <- findInterval(x$x.0.m.cx, x.bin)
    intY <- findInterval(x$x.0.m.cy, y.bin)
    for (i in seq_len(nbins))
        for (j in seq_len(nbins)) {
            ind <- which(intX == i & intY == j)
            if (length(ind) > 0) M[i, j] <- mean(x[ind, feature][[1]],
            na.rm=TRUE)
        }
    M <- flipMatrix(M)
    return(M)
}

px2area <- function(x, px) x*px*px

px2microns <- function(x, px) (x-1)*px

microns2px <- function(x, px) x/px+1

area2px <- function(x, px) x/px/px

areaLab <- expression(paste("area [", mu, m^2, "]"))

activationLab <- expression("log"[2]~"(M/C)")

timeLab <- "time [min]"

plotFeatureEvolution <- function(x, dt=32.6/60, tMax, myTitle="", cex=1.4,
    cex.axis=1, px=0.293, mar=c(5.1, 5.1, 4.1, 4.1), legend=TRUE, line=2.5)
{
    old <- par(mar=mar); on.exit(par(old))
    x$t <- x$t * dt
    ind <- x$t < tMax
    x <- x[ind, ]
    aMedian <- px2area(tapply(x$x.0.s.area, x$t, median),
        px=px)
    aQ25 <- px2area(tapply(x$x.0.s.area, x$t, quantile,
        probs=0.25), px=px)
    aQ75 <- px2area(tapply(x$x.0.s.area, x$t, quantile,
        probs=0.75), px=px)
    tSeq <- as.numeric(names(aMedian))
    shadingBox <- c(aQ75, rev(aQ25))
    shadingBoxT <- c(tSeq, rev(tSeq))
    plot(tSeq, aMedian, type="l", lwd=2, xlab=timeLab, ylab=areaLab,
        ylim=range(shadingBox), axes="F", cex.lab=cex)
    polygon(shadingBoxT, shadingBox, col=rgb(0, 0, 0.5, alpha=0.3),
        border=NA)
    axis(1, cex.axis=cex.axis, lwd=cex.axis)
    axis(2, cex.axis=cex.axis, lwd=cex.axis)
    par(new=TRUE)
    eMedian <- tapply(x$e.x, x$t, median)
    eQ25 <- tapply(x$e.x, x$t, quantile, probs=0.25)
    eQ75 <- tapply(x$e.x, x$t, quantile, probs=0.75)
    shadingBox <- c(eQ75, rev(eQ25))
    plot(tSeq, eMedian, type="l", lwd=2, xlab="", ylab="", axes="F",
        ylim=range(shadingBox), lty=2)
    polygon(shadingBoxT, shadingBox, col=rgb(0, 0.5, 0, alpha=0.3),
        border=NA)
    axis(4, cex.axis=cex.axis, lwd=cex.axis)
    mtext("a-p anisotropy", side=4, line=line, cex=cex)
    if (legend) legend("topleft", legend=c("area", "a-p anisotropy"),
        lty=1:2)
    title(myTitle)
    invisible()
}

binarizeMask <- function(x) {x[x > 1] <- 1; return(x)}

constructBox <- function(dvPos, Lx=100, Ly=50, w=512, mid=NA) {
    if (Lx > w/2) stop ("'Lx' cannot be larger 'w/2'")
    if (is.na(mid)) mid <- round(w/2)
    box <- c(xleft=mid-Lx, ybottom=dvPos-Ly, xright=mid+Lx, ytop=dvPos+Ly)
    return(box)
}

isolateBoxCells <- function(x, box) {
    ind <- x$x.0.m.cx > box["xleft"] & x$x.0.m.cx < box["xright"] &
        x$x.0.m.cy > box["ybottom"] & x$x.0.m.cy < box["ytop"]
    x <- x[ind, ]
    return(x)
}

identifyFurrowPosition <- function(x, nbinsExclude=3, h=100, plot=FALSE,
    myCex=1.4, w=512, px=0.293) {
    a <- averageFeature(x, w, "x.0.s.area", w=w)
    as <- rev(apply(a, 2, mean, na.rm=TRUE))
    myPoly <- lp(seq_along(as), h=h)
    myFit <- locfit(as ~ myPoly)
    myPred <- predict(myFit, newdata=seq_along(as))
    cropNA <- function(x, n, w) {x[c(seq_len(n), (w-n):w)] <- NA; return(x)}
    myPred <- cropNA(myPred, nbinsExclude, w=w)
    ind <- which.min(myPred)
    if (plot) {
        old <- par(mar=c(5.1, 5.1, 4.1, 4.1)); on.exit(par(old))
        seqMicrons <- function(w, px) px2microns(seq_len(w)-1, px=px)
        plot(seqMicrons(w, px), px2area(as, px=px),
            xlab=expression(paste("DV position [", mu, m, "]")), ylab=areaLab,
            cex.lab=myCex)
        points(seqMicrons(w, px), px2area(myPred, px=px), type="l", col="blue",
            lwd=2)
        abline(v=px2microns(ind, px=px), lwd=2)
    }
    return(ind)
}

identifyTimeMinArea <- function (x, h=2, px=0.293, plot=FALSE, myCex=1.4)
{
    dt <- x$dt[1]/60
    x$t <- x$t*dt
    aMedian <- px2area(tapply(x$x.0.s.area, x$t, median), px=px)
    tSeq <- as.numeric(names(aMedian))
    myPoly <- lp(tSeq, h=h)
    myFit <- locfit(aMedian ~ myPoly)
    myPred <- predict(myFit, newdata=tSeq)
    ind <- which.min(myPred)
    tstar <- tSeq[which.min(myPred)]
    if (plot) {
        old <- par(mar=c(5.1, 5.1, 4.1, 4.1)); on.exit(par(old))
        plot(tSeq, aMedian, xlab=timeLab, ylab=areaLab, cex.lab=myCex)
        points(tSeq, myPred, type="l", col="blue", lwd=2)
        abline(v=tstar, lty=2)
    }
    return(c(tstar=tstar, tindex=ind))
}

