% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculateCellDistancesSimilarity.R
\name{calculateCellDistancesSimilarity}
\alias{calculateCellDistancesSimilarity}
\title{Function to Calculate Bhattacharyya Coefficients and Hellinger Distances}
\usage{
calculateCellDistancesSimilarity(
  query_data,
  reference_data,
  query_cell_type_col,
  ref_cell_type_col,
  cell_types = NULL,
  cell_names_query,
  pc_subset = 1:5,
  assay_name = "logcounts",
  max_cells_ref = 5000
)
}
\arguments{
\item{query_data}{A \code{\linkS4class{SingleCellExperiment}} object containing numeric expression matrix for the query cells.}

\item{reference_data}{A \code{\linkS4class{SingleCellExperiment}} object containing numeric expression matrix for the reference cells.}

\item{query_cell_type_col}{The column name in the \code{colData} of \code{query_data}
that identifies the cell types.}

\item{ref_cell_type_col}{The column name in the \code{colData} of \code{reference_data}
that identifies the cell types.}

\item{cell_types}{A character vector specifying the cell types to include in the plot. If NULL, all cell types are
included.}

\item{cell_names_query}{A character vector specifying the names of the query cells for which to compute distance measures.}

\item{pc_subset}{A numeric vector specifying which principal components to include in the plot. Default is 1:5.}

\item{assay_name}{Name of the assay on which to perform computations. Default is "logcounts".}

\item{max_cells_ref}{Maximum number of reference cells to retain after cell type filtering. If NULL,
no downsampling of reference cells is performed. Default is 5000.}
}
\value{
A list containing distance data for each cell type. Each entry in the list contains:
\describe{
  \item{ref_distances}{A vector of all pairwise distances within the reference subset for the cell type.}
  \item{query_to_ref_distances}{A matrix of distances from each query cell to all reference cells for the cell type.}
}
}
\description{
This function computes Bhattacharyya coefficients and Hellinger distances to quantify the similarity of density
distributions between query cells and reference data for each cell type.
}
\details{
This function first computes distance data using the \code{calculateCellDistances} function, which calculates
pairwise distances between cells within the reference data and between query cells and reference cells in the PCA space.
Bhattacharyya coefficients and Hellinger distances are calculated to quantify the similarity of density distributions between query
cells and reference data for each cell type. Bhattacharyya coefficient measures the similarity of two probability distributions,
while Hellinger distance measures the distance between two probability distributions.

Bhattacharyya coefficients range between 0 and 1. A value closer to 1 indicates higher similarity between distributions, while a value
closer to 0 indicates lower similarity

Hellinger distances range between 0 and 1. A value closer to 0 indicates higher similarity between distributions, while a value
closer to 1 indicates lower similarity.
}
\examples{
# Load data
data("reference_data")
data("query_data")

# Plot the PC data
distance_data <- calculateCellDistances(query_data = query_data,
                                        reference_data = reference_data,
                                        query_cell_type_col = "SingleR_annotation",
                                        ref_cell_type_col = "expert_annotation",
                                        pc_subset = 1:10)

# Identify outliers for CD4
cd4_anomalies <- detectAnomaly(reference_data = reference_data,
                               query_data = query_data,
                               query_cell_type_col = "SingleR_annotation",
                               ref_cell_type_col = "expert_annotation",
                               pc_subset = 1:10,
                               n_tree = 500,
                               anomaly_threshold = 0.5)
cd4_top6_anomalies <- names(sort(cd4_anomalies$CD4$query_anomaly_scores, decreasing = TRUE)[1:6])

# Get overlap measures
overlap_measures <- calculateCellDistancesSimilarity(query_data = query_data,
                                                     reference_data = reference_data,
                                                     cell_names_query = cd4_top6_anomalies,
                                                     query_cell_type_col = "SingleR_annotation",
                                                     ref_cell_type_col = "expert_annotation",
                                                     pc_subset = 1:10)
overlap_measures

}
\author{
Anthony Christidis, \email{anthony-alexander_christidis@hms.harvard.edu}
}
