#' Summarise differential analysis results 
#' 
#' @description Print a summary of the statistical analysis of sRNA clusters 
#' (sRNAseq) or mRNAs (mRNAseq) from a mobileRNA analysis
#'
#' @param data data.frame; generated by [mobileRNA::RNAdifferentialAnalysis()]  
#'
#' 
#' @param alpha numeric; user-defined numeric value to represent the adjusted 
#' p-value threshold to define statistical significance. Defaults setting 
#' `alpha=0.1`. 
#'
#'@param chimeric logical; state whether system is chimeric: contains multiple 
#'genomes/genotypes. 
#'
#'@param controls character; vector of control condition sample names. 
#'
#'
#'@param genome.ID character; chromosome identifier of the genome representing 
#'either the origin of mobile molecules or the other genome in the chimeric 
#'system.  
#'@details 
#' To look only at the differential abundance from RNAs in the mobilome, use the 
#' `chimeric=TRUE` parameter and supply the chromosome identifier of the genome 
#' from which mobile molecules originate from to the `genome.ID` parameter & 
#' the control condition samples names to the `controls` parameter. 
#' 
#' @return Prints a summary of the RNAs which align with the 
#' adjusted p-value cutoff and states the number which has a positive and 
#' negative log-fold change. Where a positive log-fold change represents an 
#' increase in abundance and a negative value represents a decrease in abundance
#' between the conditions. 
#'  
#' @examples
#' 
#' # load data 
#' data("sRNA_data")
#' 
#' 
#'# sample conditions.
#'groups <- c("Selfgraft", "Selfgraft", "Selfgraft", "Heterograft", "Heterograft", "Heterograft")
#'
#'
#'## Differential analysis: DEseq2 method
#'sRNA_DESeq2 <- RNAdifferentialAnalysis(data = sRNA_data,
#'                              group = groups,
#'                              method = "DESeq2" )
#'                           
#'                    
#' res <- RNAsummary(sRNA_DESeq2)
#'                                  
#' @export
#' @importFrom stats complete.cases
RNAsummary <- function(data, alpha = 0.1, chimeric = FALSE,
                                  controls = NULL, genome.ID = NULL) {
  # Check if 'data' is a data frame
  if (!is.data.frame(data)) {
    stop("Input 'data' must be a data frame.")
  }
  
  # Check if 'log2FoldChange' column exists in the data frame
  if (!"log2FoldChange" %in% colnames(data)) {
    stop("Column 'log2FoldChange' not found in the data frame.
         Please refer to the vignette.")
  }
  # Check if 'padjusted' column exists in the data frame
  if (!"padjusted" %in% colnames(data)) {
    stop("Column 'padjusted' not found in the data frame.
         Please refer to the vignette.")
  }
  if(chimeric){
    data <- .remove_mapping_errors_V2(data = data,
                                      controls = controls, 
                                      genome.ID = genome.ID)
  } 
  # calculate number of non-zero rows
  total_rows <- nrow(data)
  # Filter the data based on the padjusted value
  filtered_data <- data[data$padjusted < alpha, ]
# remove any NA values 
  filtered_data <- filtered_data[stats::complete.cases(filtered_data$padjusted), ]

  # number of rows with positive and negative log2FC
  n_up <- sum(filtered_data[, "log2FoldChange"] > 0)
  n_down <- sum(filtered_data[, "log2FoldChange"] < 0)
  total_rows2 <- nrow(filtered_data)
  # Calculate percentages
  percent_up <- (n_up / total_rows2) * 100
  percent_down <- (n_down / total_rows2) * 100
  
  result_text <- c(
    paste("Comparison :" ,unique(filtered_data$comparision.order), "\n"),
    paste("out of", total_rows, "with nonzero read count \n"),
    paste("adjusted p-value <", alpha, "\n"),
    paste(total_rows2, "sRNA clusters met the adjusted p-value cutoff \n"),
    paste("LFC > 0 (higher) :", n_up, ",", round(percent_up, 1), "%", "\n"),
    paste("LFC < 0 (lower)  :", n_down, ",", round(percent_down, 1), "%", "\n"))
  
  cat(result_text, collapse = " ")
  #return()
}


