% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.mint.splsda.R
\name{tune.mint.splsda}
\alias{tune.mint.splsda}
\title{Estimate the parameters of mint.splsda method}
\usage{
tune.mint.splsda(
  X,
  Y,
  ncomp = 1,
  study,
  test.keepX = NULL,
  already.tested.X,
  scale = TRUE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  signif.threshold = 0.01,
  dist = c("max.dist", "centroids.dist", "mahalanobis.dist"),
  measure = c("BER", "overall"),
  auc = FALSE,
  progressBar = FALSE,
  light.output = TRUE
)
}
\arguments{
\item{X}{numeric matrix of predictors. \code{NA}s are allowed.}

\item{Y}{Outcome. Numeric vector or matrix of responses (for multi-response
models)}

\item{ncomp}{Number of components to include in the model (see Details).
Default to 1}

\item{study}{grouping factor indicating which samples are from the same
study}

\item{test.keepX}{numeric vector for the different number of variables to
test from the \eqn{X} data set. If set to NULL only number of components will be tuned.}

\item{already.tested.X}{if \code{ncomp > 1} Numeric vector indicating the
number of variables to select from the \eqn{X} data set on the firsts
components}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{tol}{Convergence stopping value.}

\item{max.iter}{integer, the maximum number of iterations.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Default value is FALSE}

\item{signif.threshold}{numeric between 0 and 1 indicating the significance
threshold required for improvement in error rate of the components. Default
to 0.01.}

\item{dist}{only applies to an object inheriting from \code{"plsda"} or
\code{"splsda"} to evaluate the classification performance of the model.
Should be a subset of \code{"max.dist"}, \code{"centroids.dist"},
\code{"mahalanobis.dist"}. Default is \code{"all"}. See
\code{\link{predict}}.}

\item{measure}{Two misclassification measure are available: overall
misclassification error \code{overall} or the Balanced Error Rate \code{BER}. 
Only used when \code{test.keepX = NULL}.}

\item{auc}{if \code{TRUE} calculate the Area Under the Curve (AUC)
performance of the model.}

\item{progressBar}{by default set to \code{TRUE} to output the progress bar
of the computation.}

\item{light.output}{if set to FALSE, the prediction/classification of each
sample for each of \code{test.keepX} and each comp is returned.}
}
\value{
The returned value is a list with components:
\item{error.rate}{returns the prediction error for each \code{test.keepX} on
each component, averaged across all repeats and subsampling folds. Standard
deviation is also output. All error rates are also available as a list.}
\item{choice.keepX}{returns the number of variables selected (optimal keepX)
on each component.} \item{choice.ncomp}{returns the optimal number of
components for the model fitted with \code{$choice.keepX} }
\item{error.rate.class}{returns the error rate for each level of \code{Y}
and for each component computed with the optimal keepX}

\item{predict}{Prediction values for each sample, each \code{test.keepX} and
each comp.} \item{class}{Predicted class for each sample, each
\code{test.keepX} and each comp.}

If \code{test.keepX = NULL}, returns:
\item{study.specific.error}{A list that gives BER, overall error rate and
error rate per class, for each study} \item{global.error}{A list that gives
BER, overall error rate and error rate per class for all samples}
\item{predict}{A list of length \code{ncomp} that produces the predicted
values of each sample for each class} \item{class}{A list which gives the
predicted class of each sample for each \code{dist} and each of the
\code{ncomp} components. Directly obtained from the \code{predict} output.}
\item{auc}{AUC values} \item{auc.study}{AUC values for each study in mint
models}.
}
\description{
Computes Leave-One-Group-Out-Cross-Validation (LOGOCV) scores on a
user-input grid to determine optimal values for the parameters in
\code{mint.splsda}.
}
\details{
This function performs a Leave-One-Group-Out-Cross-Validation (LOGOCV),
where each of \code{study} is left out once.

When \code{test.keepX} is not NULL, all component \eqn{1:\code{ncomp}} are tuned to identify number of variables to keep,
except the first ones for which a \code{already.tested.X} is provided. See examples below.

The function outputs the optimal number of components that achieve the best
performance based on the overall error rate or BER. The assessment is
data-driven and similar to the process detailed in (Rohart et al., 2016),
where one-sided t-tests assess whether there is a gain in performance when
adding a component to the model. Our experience has shown that in most case,
the optimal number of components is the number of categories in \code{Y} -
1, but it is worth tuning a few extra components to check (see our website
and case studies for more details).

BER is appropriate in case of an unbalanced number of samples per class as
it calculates the average proportion of wrongly classified samples in each
class, weighted by the number of samples in each class. BER is less biased
towards majority classes during the performance assessment.

More details about the prediction distances in \code{?predict} and the
supplemental material of the mixOmics article (Rohart et al. 2017).
}
\examples{
# set up data
data(stemcells)
data <- stemcells$gene
type.id <- stemcells$celltype
exp <- stemcells$study

# tune number of components
tune_res <- tune.mint.splsda(X = data,Y = type.id, ncomp=5, 
                             near.zero.var=FALSE,
                             study=exp,
                             test.keepX = NULL)

plot(tune_res)
tune_res$choice.ncomp # 1 component

## tune number of variables to keep
tune_res <- tune.mint.splsda(X = data,Y = type.id, ncomp = 1, 
                             near.zero.var = FALSE,
                             study=exp,
                             test.keepX=seq(1,10,1))

plot(tune_res)
tune_res$choice.keepX # 9 variables to keep on component 1

## only tune component 3 and keeping 10 genes on comp1
tune_res <- tune.mint.splsda(X = data, Y = type.id, ncomp = 2, study = exp,
                             already.tested.X = c(9),
                             test.keepX = seq(1,10,1))
plot(tune_res)
tune_res$choice.keepX # 10 variables to keep on comp2
}
\references{
Rohart F, Eslami A, Matigian, N, Bougeard S, Lê Cao K-A (2017).
MINT: A multivariate integrative approach to identify a reproducible
biomarker signature across multiple experiments and platforms. BMC
Bioinformatics 18:128.

mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{mint.splsda}} and http://www.mixOmics.org for more
details.
}
\author{
Florian Rohart, Al J Abadi
}
\keyword{dplot}
\keyword{multivariate}
