% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mint.plsda.R
\name{mint.plsda}
\alias{mint.plsda}
\title{P-integration with Projection to Latent Structures models (PLS) with
Discriminant Analysis}
\usage{
mint.plsda(
  X,
  Y,
  ncomp = 2,
  study,
  scale = TRUE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  all.outputs = TRUE,
  verbose.call = FALSE
)
}
\arguments{
\item{X}{numeric matrix of predictors combining multiple independent studies
on the same set of predictors. \code{NA}s are allowed.}

\item{Y}{A factor or a class vector indicating the discrete outcome of each
sample.}

\item{ncomp}{Positive Integer. The number of components to include in the
model. Default to 2.}

\item{study}{Factor, indicating the membership of each sample to each of the
studies being combined}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}

\item{verbose.call}{Logical (Default=FALSE), if set to TRUE then the \code{$call}
component of the returned object will contain the variable values for all 
parameters. Note that this may cause large memory usage.}
}
\value{
\code{mint.plsda} returns an object of class \code{"mint.plsda",
"plsda"}, a list that contains the following components:

\item{X}{the centered and standardized original predictor matrix.}
\item{Y}{original factor} \item{ind.mat}{the centered and standardized
original response vector or matrix.} \item{ncomp}{the number of components
included in the model.} \item{study}{The study grouping factor}
\item{mode}{the algorithm used to fit the model.} \item{variates}{list
containing the variates of X - global variates.} \item{loadings}{list
containing the estimated loadings for the variates - global loadings.}
\item{variates.partial}{list containing the variates of X relative to each
study - partial variates.} \item{loadings.partial}{list containing the
estimated loadings for the partial variates - partial loadings.}
\item{names}{list containing the names to be used for individuals and
variables.} \item{nzv}{list containing the zero- or near-zero predictors
information.} \item{iter}{Number of iterations of the algorithm for each
component} \item{prop_expl_var}{Percentage of explained variance for
each component and each study after setting possible missing values to zero
(note that contrary to PCA, this amount may not decrease as the aim of the
method is not to maximise the variance, but the covariance between X and the
dummy matrix Y).}
\item{call}{if \code{verbose.call = FALSE}, then just the function call is returned.
If \code{verbose.call = TRUE} then all the inputted values are accessable via
this component}
}
\description{
Function to combine multiple independent studies measured on the same
variables or predictors (P-integration) using variants of multi-group PLS-DA
for supervised classification.
}
\details{
\code{mint.plsda} function fits a vertical PLS-DA models with \code{ncomp}
components in which several independent studies measured on the same
variables are integrated. The aim is to classify the discrete outcome
\code{Y}. The \code{study} factor indicates the membership of each sample in
each study. We advise to only combine studies with more than 3 samples as
the function performs internal scaling per study, and where all outcome
categories are represented.

\code{X} can contain missing values. Missing values are handled by being
disregarded during the cross product computations in the algorithm
\code{mint.plsda} without having to delete rows with missing data.
Alternatively, missing data can be imputed prior using the
\code{impute.nipals} function.

The type of deflation used is \code{'regression'} for discriminant algorithms.
i.e. no deflation is performed on Y.

Useful graphical outputs are available, e.g. \code{\link{plotIndiv}},
\code{\link{plotLoadings}}, \code{\link{plotVar}}.
}
\examples{

data(stemcells)

res = mint.plsda(X = stemcells$gene, Y = stemcells$celltype, ncomp = 3,
study = stemcells$study)

plotIndiv(res)

#plot study-specific outputs for all studies
plotIndiv(res, study = "all.partial")

\dontrun{
#plot study-specific outputs for study "2"
plotIndiv(res, study = "2", col = 1:3, legend = TRUE)
}

}
\references{
Rohart F, Eslami A, Matigian, N, Bougeard S, Lê Cao K-A (2017).
MINT: A multivariate integrative approach to identify a reproducible
biomarker signature across multiple experiments and platforms. BMC
Bioinformatics 18:128.

Eslami, A., Qannari, E. M., Kohler, A., and Bougeard, S. (2014). Algorithms
for multi-group PLS. J. Chemometrics, 28(3), 192-201.

mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{spls}}, \code{\link{summary}}, \code{\link{plotIndiv}},
\code{\link{plotVar}}, \code{\link{predict}}, \code{\link{perf}},
\code{\link{mint.pls}}, \code{\link{mint.spls}}, \code{\link{mint.splsda}}
and http://www.mixOmics.org/mixMINT for more details.
}
\author{
Florian Rohart, Kim-Anh Lê Cao, Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
