#' Filter the database based on user inputs
#' @importFrom rlang .data
#' @param input Shiny input object
#' @return Filtered dataframe
#' @noRd
.filter_data <- function(input) {
    data <- db

    if (!is.null(input$project) && length(input$project) > 0) {
        data <- data %>% filter(.data$Project.ID %in% input$project)
    }

    if (!is.null(input$sample_type) && length(input$sample_type) > 0) {
        data <- data %>% filter(.data$Sample.Type %in% input$sample_type)
    }

    if (!is.null(input$source_site) && length(input$source_site) > 0) {
        data <- data %>% filter(.data$Source.Site %in% input$source_site)
    }

    if (!is.null(input$state) && length(input$state) > 0) {
        data <- data %>% filter(.data$state %in% input$state)
    }

    if (!is.null(input$case_search) && input$case_search != "") {
        data <- data %>%
            filter(grepl(input$case_search, .data$Case.ID, ignore.case = TRUE))
    }

    data
}

#' Prepare geographic data for visualization
#' @param data Filtered data
#' @return Processed geographic data
#' @noRd
.prepare_geo_data <- function(data) {
    data %>%
        group_by(.data$Source.Site, .data$lat, .data$lon, .data$state) %>%
        summarize(
            samples = n(),
            cases = n_distinct(.data$Case.ID),
            .groups = 'drop'
        )
}

#' Prepare dotplot data
#' @param data Filtered data
#' @param input Shiny input object
#' @return Processed dotplot data with attributes
#' @noRd
.prepare_dotplot_data <- function(data, input) {
    result <- data %>%
        count(!!sym(input$dotplot_x), !!sym(input$dotplot_y)) %>%
        rename(
            Var1 = !!sym(input$dotplot_y),
            Var2 = !!sym(input$dotplot_x),
            Freq = n
        )

    attr(result, "x_label") <- input$dotplot_x
    attr(result, "y_label") <- input$dotplot_y

    result
}

#' Generate download code based on filtered data
#' @param data Filtered data
#' @return Character string containing R code
#' @noRd
.generate_download_code <- function(data) {
    file_ids <- paste(sprintf('"%s"', data$File.ID), collapse = ",\n  ")

    sprintf('## Make sure BiocManager is installed
if (!require("BiocManager", quietly = TRUE))
    install.packages("BiocManager")

if (!require("GenomicDataCommons", quietly = TRUE))
    BiocManager::install("GenomicDataCommons")
library("GenomicDataCommons")

# File IDs to download
file_ids <- c(
%s
)

# Download files
lapply(file_ids, gdcdata)', file_ids)
}

#' Get selected rows from data table
#' @param input Shiny input object
#' @return Selected data rows
#' @noRd
.get_selected_rows <- function(input) {
    s <- input$data_table_rows_selected
    if (is.null(s)) return(NULL)
    .filter_data(input)[s, ]
}
