% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quality.R
\name{add_quality_flags}
\alias{add_quality_flags}
\alias{h_low_expression_flag}
\alias{h_low_depth_flag}
\alias{h_tech_failure_flag}
\alias{get_tech_failure}
\alias{get_low_depth}
\alias{get_low_expression}
\title{Add Quality Flags}
\usage{
add_quality_flags(object, control = control_quality(), overwrite = FALSE)

h_low_expression_flag(object, control = control_quality())

h_low_depth_flag(object, control = control_quality())

h_tech_failure_flag(object, control = control_quality())

get_tech_failure(object)

get_low_depth(object)

get_low_expression(object)
}
\arguments{
\item{object}{(\code{AnyHermesData}) \cr input.}

\item{control}{(\code{list}) \cr list of settings (thresholds etc.) used to compute the
quality control flags, produced by \code{\link[=control_quality]{control_quality()}}.}

\item{overwrite}{(\code{flag})\cr whether previously added flags may be overwritten.}
}
\value{
The input object with added quality flags.
}
\description{
The function \code{add_quality_flags()} adds quality flag information to a \code{\link{AnyHermesData}} object:
\itemize{
\item \code{low_expression_flag}: for each gene, counts how many samples don't pass a minimum
expression Counts per Million (\code{CPM}) threshold. If too many, then it flags this gene
as a "low expression" gene.
\item \code{tech_failure_flag}: first calculates the Pearson correlation matrix of the sample wise
\code{CPM} values, resulting in a matrix measuring the correlation between samples.
Then compares the average correlation per sample with a threshold - if it is too low,
then the sample is flagged as a "technical failure" sample.
\item \code{low_depth_flag}: computes the library size (total number of counts) per sample.
If this number is too low, the sample is flagged as a "low depth" sample.
}

Separate helper functions are internally used to create the flags, and
separate \code{getter} functions allow easy access to the quality control flags in an object.
}
\details{
While \code{object} already has the variables mentioned above as part of the
\code{rowData} and \code{colData} (as this is enforced by the validation
method for \code{\link{AnyHermesData}}), they are usually still \code{NA} after the initial
object creation.
}
\section{Functions}{
\itemize{
\item \code{h_low_expression_flag()}: creates the low expression flag for genes
given control settings.

\item \code{h_low_depth_flag()}: creates the low depth (library size) flag for samples
given control settings.

\item \code{h_tech_failure_flag()}: creates the technical failure flag for samples
given control settings.

\item \code{get_tech_failure()}: get the technical failure flags for all samples.

\item \code{get_low_depth()}: get the low depth failure flags for all samples.

\item \code{get_low_expression()}: get the low expression failure flags for all genes.

}}
\examples{
# Adding default quality flags to `AnyHermesData` object.
object <- hermes_data
result <- add_quality_flags(object)
which(get_tech_failure(result) != get_tech_failure(object))
head(get_low_expression(result))
head(get_tech_failure(result))
head(get_low_depth(result))

# It is possible to overwrite flags if needed, which will trigger a message.
result2 <- add_quality_flags(result, control_quality(min_cpm = 1000), overwrite = TRUE)

# Separate calculation of low expression flag.
low_expr_flag <- h_low_expression_flag(
  object,
  control_quality(min_cpm = 500, min_cpm_prop = 0.9)
)
length(low_expr_flag) == nrow(object)
head(low_expr_flag)

# Separate calculation of low depth flag.
low_depth_flag <- h_low_depth_flag(object, control_quality(min_depth = 5))
length(low_depth_flag) == ncol(object)
head(low_depth_flag)

# Separate calculation of technical failure flag.
tech_failure_flag <- h_tech_failure_flag(object, control_quality(min_corr = 0.35))
length(tech_failure_flag) == ncol(object)
head(tech_failure_flag)
head(get_tech_failure(object))
head(get_low_depth(object))
head(get_low_expression(object))
}
\seealso{
\itemize{
\item \code{\link[=control_quality]{control_quality()}} for the detailed settings specifications;
\item \code{\link[=set_tech_failure]{set_tech_failure()}} to manually flag samples as technical failures.
}
}
