#' Sequentially apply a manual gating strategy to a GatingSet or list
#'
#' This function allows for a "semi-automatic" approach to interactive manual
#' gating of flow cytometry data. It leverages the purrr package to let you
#' easily define a gating strategy and then apply it sequentially to a
#' GatingSet. This will call gs_gate_interactive() once for each line in your
#' gating strategy, applying it to your GatingSet as soon as you draw each
#' prompted gate.
#'
#' The gating strategy should be a tibble, with each column name corresponding
#' to one parameter from gs_gate_interactive. Any parameters not specified in
#' this tibble will either use their defaults from gs_gate_interactive or can be
#' specified directly in the function call to gs_apply_gating_strategy.
#' Typically, this gating strategy will have a column for 'filterId', 'dims',
#' 'subset', and 'coords', but techinicaly only filterId is required. See
#' examples below for an easy way to construct this strategy using tribble().
#'
#'
#' @param gs A GatingSet or list of GatingSets.
#' @param gating_strategy A tibble-formatted gating strategy (see examples below)
#' @param ... Other parameters to pass to gs_gate_interactive(). Note that only
#'   constant parameters should be supplied here---anything that varies should
#'   be included in the gating_strategy tibble.
#'
#' @return the GatingSet or list of GatingSets with the gates in gating_strategy
#'   applied as specified.
#'
#' @importFrom tibble tribble
#' @importFrom methods is
#'
#' @examples
#'
#' fs <- flowCore::read.flowSet(
#'   path = system.file("extdata", package = "flowGate"), pattern = ".FCS$")
#'
#' gs <- flowWorkspace::GatingSet(fs)
#'
#' # Note - this is a very rudamentary GatingSet for example purposes only. 
#' # Please see the vignette accompanying this package or the flowWorkspace
#' # documentation # for a complete look at creating a GatingSet.
#'
#' gating_strategy <- tibble::tribble(
#' ~filterId,      ~dims,                       ~subset,        ~coords,
#' "Lymphocytes", list("FSC-H", "SSC-H"),       "root",         list(c(0, 3e5), c(0, 3e5)),
#' "CD45 CD15",   list("CD45 PE", "CD15 FITC"), "Lymphocytes",  list(c(0, 3e5), c(0, 2e5)),
#' )
#'
#'
#' if(interactive()){
#' gs_apply_gating_strategy(gs,
#' gating_strategy = gating_strategy,
#' bins = 512) # note that extra args for gs_gate_interactive can be supplied.
#' }
#' @export
gs_apply_gating_strategy <- function(gs, gating_strategy, ...){
    if(methods::is(gs, "GatingSet")){
        purrr::pmap(gating_strategy, 
                    flowGate::gs_gate_interactive, gs = gs, ...)
    } else {
        stop("'gs' must be a GatingSet")
    }
}
