% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regionMatrix.R
\name{regionMatrix}
\alias{regionMatrix}
\title{Identify regions data by a coverage filter and get a count matrix}
\usage{
regionMatrix(
  fullCov,
  cutoff = 5,
  L,
  totalMapped = 8e+07,
  targetSize = 8e+07,
  runFilter = TRUE,
  returnBP = TRUE,
  ...
)
}
\arguments{
\item{fullCov}{A list where each element is the result from
\link{loadCoverage} used with \code{returnCoverage = TRUE}. Can be generated
using \link{fullCoverage}. If \code{runFilter = FALSE}, then
\code{returnMean = TRUE} must have been used.}

\item{cutoff}{The base-pair level cutoff to use. It's behavior is controlled
by \code{filter}.}

\item{L}{The width of the reads used. Either a vector of length 1 or length
equal to the number of samples.}

\item{totalMapped}{A vector with the total number of reads mapped for each
sample. The vector should be in the same order as the samples in
\code{fullCov}. Providing this argument adjusts the coverage to reads in
\code{targetSize} library prior to filtering. See \link{getTotalMapped} for
calculating this vector.}

\item{targetSize}{The target library size to adjust the coverage to. Used
only when \code{totalMapped} is specified. By default, it adjusts to
libraries with 80 million reads.}

\item{runFilter}{This controls whether to run \link{filterData} or not. If
set to \code{FALSE} then \code{returnMean = TRUE} must have been used to
create each element of \code{fullCov} and the data must have been
normalized (\code{totalMapped} equal to \code{targetSize}).}

\item{returnBP}{If \code{TRUE}, returns \verb{$bpCoverage} explained below.}

\item{...}{Arguments passed to other methods and/or advanced arguments.
Advanced arguments:
\describe{
\item{verbose }{ If \code{TRUE} basic status updates will be printed along
the way.}
\item{chrsStyle }{ Default: \code{UCSC}. Passed to
\link{extendedMapSeqlevels} via \link{getRegionCoverage}.}
\item{species }{ Default: \code{homo_sapiens}. Passed to
\link{extendedMapSeqlevels} via \link{getRegionCoverage}.}
\item{currentStyle }{ Default: \code{NULL}. Passed to
\link{extendedMapSeqlevels} via \link{getRegionCoverage}.}
}
Passed to \link{filterData}, \link{findRegions} and \link{define_cluster}.

Note that \link{filterData} is used internally
by \link{loadCoverage} (and hence \code{fullCoverage}) and has the important
arguments \code{totalMapped} and \code{targetSize} which can be used to
normalize the coverage by library size. If you already used these arguments
when creating the \code{fullCov} object, then don't specify them a second
time in \link{regionMatrix}. If you have not used these arguments, we
recommend using them to normalize the mean coverage.}
}
\value{
A list with one entry per chromosome. Then per chromosome, a list
with three components.
\describe{
\item{regions }{ A set of regions based on the coverage filter cutoff as
returned by \link{findRegions}.}
\item{bpCoverage }{ A list with one element per region. Each element is a matrix with numbers of rows equal to the number of base pairs in the region and number of columns equal to the number of samples. It contains the base-level coverage information for the regions. Only returned when \code{returnBP = TRUE}.}
\item{coverageMatrix }{  A matrix with the mean coverage by sample for each
candidate region.}
}
}
\description{
Given a set of un-filtered coverage data (see \link{fullCoverage}), create
candidate regions by applying a cutoff on the coverage values,
and obtain a count matrix where the number of rows corresponds to the number
of candidate regions and the number of columns corresponds to the number of
samples. The values are the mean coverage for a given sample for a given
region.
}
\details{
This function uses several other \link{derfinder-package}
functions. Inspect the code if interested.

You should use at most one core per chromosome.
}
\examples{
## Create some toy data
library("IRanges")
x <- Rle(round(runif(1e4, max = 10)))
y <- Rle(round(runif(1e4, max = 10)))
z <- Rle(round(runif(1e4, max = 10)))
fullCov <- list("chr21" = DataFrame(x, y, z))

## Calculate a proxy of library size
libSize <- sapply(fullCov$chr21, sum)

## Run region matrix normalizing the coverage
regionMat <- regionMatrix(
    fullCov = fullCov, maxRegionGap = 10L,
    maxClusterGap = 300L, L = 36, totalMapped = libSize, targetSize = 4e4
)
\dontrun{
## You can alternatively use filterData() on fullCov to reduce the required
## memory before using regionMatrix(). This can be useful when mc.cores > 1
filteredCov <- lapply(fullCov, filterData,
    returnMean = TRUE, filter = "mean",
    cutoff = 5, totalMapped = libSize, targetSize = 4e4
)
regionMat2 <- regionMatrix(filteredCov,
    maxRegionGap = 10L,
    maxClusterGap = 300L, L = 36, runFilter = FALSE
)
}

## regionMatrix() can work with multiple chrs as shown below.
fullCov2 <- list("chr21" = DataFrame(x, y, z), "chr22" = DataFrame(x, y, z))
regionMat2 <- regionMatrix(
    fullCov = fullCov2, maxRegionGap = 10L,
    maxClusterGap = 300L, L = 36, totalMapped = libSize, targetSize = 4e4
)

## Combine results from multiple chromosomes
library("GenomicRanges")

## First extract the data
regs <- unlist(GRangesList(lapply(regionMat2, "[[", "regions")))
covMat <- do.call(rbind, lapply(regionMat2, "[[", "coverageMatrix"))
covBp <- do.call(c, lapply(regionMat2, "[[", "bpCoverage"))
## Force the names to match
names(regs) <- rownames(covMat) <- names(covBp) <- seq_len(length(regs))
## Combine into a list (not really needed)
mergedRegionMat <- list(
    "regions" = regs, "coverageMatrix" = covMat,
    "bpCoverage" = covBp
)
}
\author{
Leonardo Collado-Torres
}
