% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findRegions.R
\name{findRegions}
\alias{findRegions}
\title{Find non-zero regions in a Rle}
\usage{
findRegions(
  position = NULL,
  fstats,
  chr,
  oneTable = TRUE,
  maxClusterGap = 300L,
  cutoff = quantile(fstats, 0.99, na.rm = TRUE),
  segmentIR = NULL,
  smooth = FALSE,
  weights = NULL,
  smoothFunction = bumphunter::locfitByCluster,
  ...
)
}
\arguments{
\item{position}{A logical Rle of genomic positions. This is generated in
\link{loadCoverage}. Note that it gets updated in \link{preprocessCoverage}
if \code{colsubset} is not \code{NULL}.}

\item{fstats}{A numeric Rle with the F-statistics. Usually obtained using
\link{calculateStats}.}

\item{chr}{A single element character vector specifying the chromosome name.}

\item{oneTable}{If \code{TRUE} only one GRanges is returned.
Otherwise, a GRangesList with two components is returned: one for the
regions with positive values and one for the negative values.}

\item{maxClusterGap}{This determines the maximum gap between candidate DERs.
It should be greater than \code{maxRegionGap} (0 by default).}

\item{cutoff}{Threshold applied to the \code{fstats} used to determine the
regions.}

\item{segmentIR}{An IRanges object with the genomic positions that are
potentials DERs. This is used in \link{calculatePvalues} to speed up
permutation calculations.}

\item{smooth}{Whether to smooth the F-statistics (\code{fstats}) or not. This
is by default \code{FALSE}. For RNA-seq data we recommend using \code{FALSE}.}

\item{weights}{Weights used by the smoother as described in
\link[bumphunter:smoother]{smoother}.}

\item{smoothFunction}{A function to be used for smoothing the F-statistics.
Two functions are provided by the \code{bumphunter} package:
\link[bumphunter:loessByCluster]{loessByCluster} and \link[bumphunter:runmedByCluster]{runmedByCluster}. If
you are using your own custom function, it has to return a named list with
an element called \verb{$fitted} that contains the smoothed F-statistics and
an element claled \verb{$smoothed} that is a logical vector indicating
whether the F-statistics were smoothed or not. If they are not smoothed, the
original values will be used.}

\item{...}{Arguments passed to other methods and/or advanced arguments.
Advanced arguments:
\describe{
\item{verbose }{ If \code{TRUE} basic status updates will be printed along
the way.}
\item{basic }{ If \code{TRUE} a DataFrame is returned that has only basic
information on the candidate DERs. This is used in \link{calculatePvalues}
to speed up permutation calculations. Default: \code{FALSE}.}
\item{maxRegionGap }{ This determines the maximum number of gaps between two
genomic positions to be considered part of the same candidate region. The
default is 0L.}
}
Passed to \link{extendedMapSeqlevels} and the internal function
\code{.getSegmentsRle} that has by default \code{verbose = FALSE}.

When \code{smooth = TRUE}, \code{...} is passed to the internal function
\code{.smootherFstats}. This internal function has the advanced argument
\code{maxClusterGap} (same as above) and passes \code{...} to
\link{define_cluster} and the formal arguments of \code{smoothFun}.}
}
\value{
Either a GRanges or a GRangesList as determined by \code{oneTable}.
Each of them has the following metadata variables.
\describe{
\item{value }{ The mean of the values of \code{y} for the given region.}
\item{area }{  The absolute value of the sum of the values of \code{y} for
the given region.}
\item{indexStart }{ The start position of the region in terms of the index
for \code{y}.}
\item{indexEnd }{ The end position of the region in terms of the index for
\code{y}.}
\item{cluster }{ The cluser ID.}
\item{clusterL }{ The total length of the cluster.}
}
}
\description{
Find genomic regions for which a numeric vector is above (or below)
predefined thresholds. In other words, this function finds the candidate
Differentially Expressed Regions (candidate DERs). This is similar to
\link[bumphunter:regionFinder]{regionFinder} and is a helper function for
\link{calculatePvalues}.
}
\details{
\link[bumphunter:regionFinder]{regionFinder} adapted to Rle world.
}
\examples{
## Preprocess the data
prep <- preprocessCoverage(genomeData,
    cutoff = 0, scalefac = 32, chunksize = 1e3,
    colsubset = NULL
)

## Get the F statistics
fstats <- genomeFstats

## Find the regions
regs <- findRegions(prep$position, fstats, "chr21", verbose = TRUE)
regs
\dontrun{
## Once you have the regions you can proceed to annotate them
library("bumphunter")
genes <- annotateTranscripts(TxDb.Hsapiens.UCSC.hg19.knownGene::TxDb.Hsapiens.UCSC.hg19.knownGene)
annotation <- matchGenes(regs, genes)
annotation
}

# Find regions with smoothing the F-statistics by bumphunter::runmedByCluster
regs_smooth <- findRegions(prep$position, fstats, "chr21",
    verbose = TRUE,
    smoothFunction = bumphunter::runmedByCluster
)
## Compare against the object regs obtained earlier
regs_smooth
}
\references{
Rafael A. Irizarry, Martin Aryee, Hector Corrada Bravo, Kasper
D. Hansen and Harris A. Jaffee. bumphunter: Bump Hunter. R package version
1.1.10.
}
\seealso{
\link{calculatePvalues}
}
\author{
Leonardo Collado-Torres
}
