% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measureObjects.R
\name{measureObjects}
\alias{measureObjects}
\title{Compute morphological and intensity features from objects on images.}
\usage{
measureObjects(
  mask,
  image,
  img_id,
  return_as = c("sce", "spe"),
  feature_types = c("basic", "shape", "moment"),
  basic_feature = "mean",
  basic_quantiles = NULL,
  shape_feature = c("area", "radius.mean"),
  moment_feature = c("cx", "cy", "majoraxis", "eccentricity"),
  haralick_feature = NULL,
  haralick_nbins = 32,
  haralick_scales = c(1, 2),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{mask}{a \code{\linkS4class{CytoImageList}} object containing single-channel
\code{\linkS4class{Image}} or \code{\linkS4class{HDF5Array}} objects.
Segmentation masks must contain integer pixel values where groups of
pixels correspond to objects.}

\item{image}{a \code{\linkS4class{CytoImageList}} object containing single or
multi-channel \code{\linkS4class{Image}} or \code{\linkS4class{HDF5Array}}
objects, where each channel indicates the measured pixel intensities.}

\item{img_id}{character specifying the \code{mcols(image)} and
\code{mcols(mask)} entry, in which the image IDs are stored.}

\item{return_as}{single character specifying the class of the returned object.
This is either \code{"sce"} to return a \code{SingleCellExperiment} (default)
or \code{"spe"} to return a \code{SpatialExperiment} object.}

\item{feature_types}{character vector or string indicating which features to
compute. Needs to contain \code{"basic"}. Optionally, \code{"shape"},
\code{"moment"} and \code{"haralick"} are allowed. Default \code{"basic"},
\code{"shape"} and \code{"moment"}.}

\item{basic_feature}{string indicating which intensity measurement per object
and channel should be used to populate the \code{counts(x)} slot; where
\code{x} is the returned object. Default
\code{"mean"} but \code{"sd"}, \code{"mad"} and \code{"q*"} allowed. Here,
\code{*} indicates the computed quantile (see \code{basic_quantiles}).}

\item{basic_quantiles}{numeric vector or single number indicating which
quantiles to compute. Default none.}

\item{shape_feature}{string or character vector specifying which shape
features to compute. Default \code{"area"} and \code{"radius.mean"}.
Allowed entries are: \code{"area"}, \code{"perimeter"},
\code{"radius.mean"}, \code{"radius.sd"}, \code{"radius.max"},
\code{"radius.min"}.}

\item{moment_feature}{string or character vector indicating which moment
features to compute. Default \code{"cx"}, \code{"cy"}, \code{"majoraxis"},
and \code{"eccentricity"}. Other allowed features are \code{"theta"}. Here
moment features are only computed on the segmentation mask without
incorporating pixel intensities. Therefore, \code{"cx"} and \code{"cy"} are
the x and y coordinates of the cell centroids.}

\item{haralick_feature}{string or character vector indicating which haralick
features to compute. Default none. Allowed are the 13 haralick features:
\code{"asm"}, \code{"con"}, \code{"cor"}, \code{"var"}, \code{"idm"},
\code{"sav"}, \code{"sva"}, \code{"sen"}, \code{"ent"}, \code{"dva"},
\code{"den"}, \code{"f12"}, \code{"f13"}}

\item{haralick_nbins}{an integer indicating the number of bins used to
compute the haralick matrix. Pixel intensities are binned in
\code{haralick_nbins} discrete gray levels before computing the haralick
matrix.}

\item{haralick_scales}{an integer vector indicating the number of scales
(distance at which to consider neighbouring pixels) to use to compute the
haralick features.}

\item{BPPARAM}{parameters for parallelised processing of images.
See \code{\linkS4class{MulticoreParam}} for information on how to use multiple
cores for parallelised processing.}
}
\value{
A \linkS4class{SingleCellExperiment} or 
\linkS4class{SpatialExperiment} object (see details)
}
\description{
For each object (e.g. cell) identified by segmentation, the
\code{measureObjects} function computes intensity features (also referred to
as basic features; e.g. mean intensity), shape features (e.g. area), moment
features (e.g. position) and haralick features. These features are returned
in form of a \linkS4class{SingleCellExperiment} or 
\linkS4class{SpatialExperiment} object.
}
\section{The returned objects}{

By default, a \code{SingleCellExperiment} object is returned. When setting
\code{return_as = "spe"}, the returned object is of class
\code{SpatialExperiment}. The returned object contains a single assay. This
assay contains individual objects in columns and channels in rows. Each entry
summarises the intensities per object and channel. This summary statistic is
typically the mean intensity per object and channel. However, other summary
statistics can be computed. When the mean intensity per object and channel is
computed (default), the assay is accessible via \code{counts(sce)}.
Otherwise, the assay needs to be accessed via \code{assay(sce, "counts_*")},
where \code{*} indicates the argument to \code{basic_feature}.

The \code{colData(x)} entry is populated by the computed shape, moment and
haralick features per object. The prefix of the feature names indicate
whether these features correspond to shape (\code{s.}), moment (\code{m.}) or
haralick (\code{h.}) features. Default features are the following:

\itemize{
\item s.area - object size in pixels
\item s.radius.mean - mean object radius in pixels
\item m.cx - x centroid position of object
\item m.cy - y centroid position of object
\item m.majoraxis - major axis length in pixels of elliptical fit
\item m.eccentricity - elliptical eccentricity. 1 meaning straight line and 0
meaning circle.
}
}

\section{Computing quantiles}{

Sometimes it can be useful to describe the summarised pixel intensity per
object and channel not in terms of the mean but some quantile of the pixel
distribution. For example, to compute the median pixel intensity per object
and channel, set \code{basic_feature = "q05"} and \code{basic_quantiles =
0.5}.
}

\examples{
# Standard example
data(pancreasImages)
data(pancreasMasks)

sce <- measureObjects(pancreasMasks, pancreasImages, img_id = "ImageNb")
sce

# Compute only intensity feature
sce <- measureObjects(pancreasMasks, pancreasImages, img_id = "ImageNb",
                        feature_types = "basic")
colData(sce)

# Visualize on segmentation masks
plotCells(pancreasMasks, object = sce, img_id = "ImageNb",
            cell_id = "object_id", colour_by = "PIN")

}
\seealso{
\code{\link{computeFeatures}}, for detailed explanation for the computed features.
\url{https://earlglynn.github.io/RNotes/package/EBImage/Haralick-Textural-Features.html}
for more discussion on the haralick features
}
\author{
Nils Eling (\email{nils.eling@dqbm.uzh.ch}),
}
