#' Generate individual plots from countsimQCReport output
#'
#' Generate separate plots for all evaluation criteria using the collection of
#' ggplot objects that can be saved when generating a countsimQC report (by
#' setting \code{savePlots = TRUE}).
#'
#' @param ggplotsRds The path to a .rds file generated by
#'   \code{countsimQCReport} by setting \code{savePlots = TRUE}, or the list of
#'   plots stored in this file.
#' @param device One of "eps", "ps", "tex" (pictex), "pdf", "jpeg", "tiff",
#'   "png", "bmp", "svg" or "wmf" (windows only) (will be provided to the
#'   \code{ggsave} function from the \code{ggplot2} package).
#' @param outputDir The output directory where the plots should be generated.
#' @param nDatasets The number of data sets that are compared in the figures.
#'   This is needed to set the size of the plots correctly.
#'
#' @author Charlotte Soneson
#'
#' @export
#'
#' @importFrom grDevices n2mfrow
#' @importFrom methods is
#' @importFrom ggplot2 is_ggplot
#'
#' @return Nothing is returned, but plots are generated in the designated
#'   output directory.
#'
#' @examples
#' ## Load example data
#' data(countsimExample)
#' \dontrun{
#' ## Generate report
#' countsimQCReport(countsimExample, outputDir = "./",
#'                  outputFile = "example.html", savePlots = TRUE)
#' ## Generate individual plots
#' generateIndividualPlots("example_ggplots.rds", nDatasets = 3)
#' }
#'
generateIndividualPlots <- function(ggplotsRds, device = "png",
                                    outputDir = "./", nDatasets = 2) {
  ## ------------------------------------------------------------------------ ##
  ## --------------------- Check input arguments ---------------------------- ##
  ## ------------------------------------------------------------------------ ##

  if (is(ggplotsRds, "character")) {
    ggplotsRds <- readRDS(ggplotsRds)
  }
  if (!is(ggplotsRds, "list")) {
    stop("The provided ggplotsRds object, or the object stored ",
         "in the ggplotsRds file, must be a list.")
  }
  if (!all(vapply(ggplotsRds, function(w) is_ggplot(w), FALSE))) {
    stop("The elements of the provided ggplotsRds object, or the object ",
         "stored in the ggplotsRds file, must be ggplot objects. ")
  }

  ## ------------------------------------------------------------------------ ##
  ## --------------------- Create output directory -------------------------- ##
  ## ------------------------------------------------------------------------ ##

  if (!dir.exists(outputDir)) {
    dir.create(outputDir, recursive = TRUE)
  }

  ## ------------------------------------------------------------------------ ##
  ## ------------------------- Generate plots ------------------------------- ##
  ## ------------------------------------------------------------------------ ##

  colRow <- grDevices::n2mfrow(nDatasets)
  for (pl in names(ggplotsRds)) {
    if (any(vapply(c("BCVedgeR", "dispersionDESeq2", "SepHist", "SepScatter"),
                   function(x) length(grep(x, pl)) > 0, FALSE))) {
      ggsave(plot = ggplotsRds[[pl]], filename = paste0(pl, ".", device),
             path = outputDir, width = 4 * colRow[1],
             height = 4 * colRow[2], units = "in")
    } else {
      ggsave(plot = ggplotsRds[[pl]], filename = paste0(pl, ".", device),
             path = outputDir, width = 7, height = 5, units = "in")
    }
  }
}
