% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CLfdr.R
\name{CLfdr}
\alias{CLfdr}
\title{Conditional local FDR (CLfdr)}
\usage{
CLfdr(x, y, pval = NULL, pi0.method = "RB", bw.init = NULL,
    bw = NULL, reltol = 1e-04, n.subsample = NULL, check.gam = FALSE,
    k.gam = NULL, info = TRUE)
}
\arguments{
\item{x}{covariates, could be a vector of length \emph{m} or a matrix
with \emph{m} rows.}

\item{y}{a vector of \emph{z}-values of length \emph{m}.}

\item{pval}{a vector of p-values of length \emph{m}. The p-values are
only used to computed the overall true null proportion when
\code{pi0.method}="RB".}

\item{pi0.method}{method to estimate the overall true null proportion (pi0).
"RB" for the right-boundary procedure (Liang and Nettleton, 2012, JRSSB) or
"JC" (Jin and Cai, 2007, JASA).}

\item{bw.init}{initial values for bandwidth, optional. If not specified,
normal-reference rule will be used.}

\item{bw}{bandwidth values.}

\item{reltol}{relative tolerance in optim function.}

\item{n.subsample}{size of the subsample when esitmating bandwidth.}

\item{check.gam}{indicator to perform gam.check function on the
nonparametric fit.}

\item{k.gam}{tuning parameter for mgcv::gam.}

\item{info}{indicator to print out fitting information.}
}
\value{
\item{fdr}{a vector of local FDR estimates. fdr[i] is the posteiror
probability of the ith null hypothesis is true given all the data.
1-fdr[i] is the posterior probability of being a signal (the corresponding
null hypothesis is false).}
\item{FDR}{a vector of FDR values (q-values), which can be used to control
FDR at a certain level by thresholding the FDR values.}
\item{pi0}{a vector of true null probability estimates. This contains the
prior probabilities of being null.}
\item{bw}{a vector of bandwidths for conditional alternative density
estimation}
\item{fit.gam}{an object of mgcv::gam}
}
\description{
CLfdr returns the local false discovery rate (FDR) conditional on auxiliary
covariate information
}
\details{
In many multiple testing applications, the auxiliary information is widely
available and can be useful. Such information can be summary statistics
from a similar experiment or disease, the lengths of gene coding regions,
and minor allele frequencies of SNPs.

\code{y} is a vector of \emph{m} \emph{z}-values, one of each hypothesis
under test. The \emph{z}-values follow N(0,1) if their corresponding
null hypotheses are true. Other types of test statistics, such
as \emph{t}-statistics and \emph{p}-values can be transformed to
\emph{z}-values. In practice, if the distribution of \emph{z}-values is
far from N(0,1), recentering and rescaling of the \emph{z}-values
may be necessary.

\code{x} contains auxiliary covariate information. For a single covariate,
\code{x} should be a vector of length \emph{m}. For multiple covariates,
\code{x} should be a matrix with \emph{m} rows. The covariates can be either
continuous or ordered.

\code{pi0.method} specifies the method used to estimate the overall true
null proportion. If the \emph{z}-values are generated from the normal
means model, the "JC" method from Jin and Cai (2007) JASA can be
a good candidate. Otherwise, the right-boundary procedure ("RB",
Liang and Nettleton, 2012, JRSSB) is used.

\code{bw} are bandwidth values for estimating local alternative density.
Suppose there are \emph{p} covariates, then \code{bw} should be a
vector of \emph{p}+1 positive numerical values. By default, these
bandwidth values are chosen by cross-validation to minimize
a certain error measure. However, finding the optimal bandwidth
values by cross-validation can be computationally intensive,
especially when \emph{p} is not small. If good estimates of bandwidth
values are available, for example, from the analysis of a similar
dataset, the bandwidth values can be specified explicitly to save time.

\code{reltol} specifies the relative convergence tolerance when choosing the
bandwidth values (\code{bw}). It will be passed on to
\code{\link[stats:optim]{stats::optim()}}. For most analyses, the default value
of 1e-4 provides reasonably good results. A smaller value such as 1e-5
or 1e-6 could be used for further improvement at the cost of more
computation time.
}
\examples{
data(pso)
ind.nm <- is.na(pso$tval_mic)
x <- pso$len_gene[ind.nm]
# normalize covariate
x <- rank(x)/length(x)
y <- pso$zval[ind.nm]
# assign names to the z-values helps to give names to the output variables
names(y) <- row.names(pso)[ind.nm]

fit.nm <- CLfdr(x=x, y=y)
fit.nm$fdr[1:5]
}
\references{
Liang (2019), Empirical Bayes analysis of RNA sequencing
experiments with auxiliary information, to appear in Annals of Applied
Statistics
}
\author{
Kun Liang, \email{kun.liang@uwaterloo.ca}
}
