% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_AnnData.R
\name{as_AnnData}
\alias{as_AnnData}
\alias{as_AnnData.SingleCellExperiment}
\alias{as_AnnData.Seurat}
\title{Convert to an \code{AnnData} object}
\usage{
as_AnnData(
  x,
  x_mapping = NULL,
  layers_mapping = TRUE,
  obs_mapping = TRUE,
  var_mapping = TRUE,
  obsm_mapping = TRUE,
  varm_mapping = TRUE,
  obsp_mapping = TRUE,
  varp_mapping = TRUE,
  uns_mapping = TRUE,
  assay_name = NULL,
  output_class = c("InMemory", "HDF5AnnData", "ReticulateAnnData"),
  ...
)

\method{as_AnnData}{SingleCellExperiment}(
  x,
  x_mapping = NULL,
  layers_mapping = TRUE,
  obs_mapping = TRUE,
  var_mapping = TRUE,
  obsm_mapping = TRUE,
  varm_mapping = TRUE,
  obsp_mapping = TRUE,
  varp_mapping = TRUE,
  uns_mapping = TRUE,
  assay_name = TRUE,
  output_class = c("InMemory", "HDF5AnnData", "ReticulateAnnData"),
  ...
)

\method{as_AnnData}{Seurat}(
  x,
  x_mapping = NULL,
  layers_mapping = TRUE,
  obs_mapping = TRUE,
  var_mapping = TRUE,
  obsm_mapping = TRUE,
  varm_mapping = TRUE,
  obsp_mapping = TRUE,
  varp_mapping = TRUE,
  uns_mapping = TRUE,
  assay_name = NULL,
  output_class = c("InMemory", "HDF5AnnData", "ReticulateAnnData"),
  ...
)
}
\arguments{
\item{x}{The object to convert}

\item{x_mapping}{A string specifying the data to map to the \code{X} slot. If
\code{NULL}, no data will be copied to the \code{X} slot. See below for details.}

\item{layers_mapping}{A named character vector where the names are keys of
\code{layers} in the new \code{AnnData} object and values are the names of items in
the corresponding slot of \code{x}. See below for details.}

\item{obs_mapping}{A named character vector where the names are names of
\code{obs} columns in the new \code{AnnData} object and values are the names of
columns in the corresponding slot of \code{x}. See below for details.}

\item{var_mapping}{A named character vector where the names are names of
\code{var} columns in the new \code{AnnData} object and values are the names of
columns in the corresponding slot of \code{x}. See below for details.}

\item{obsm_mapping}{A named character vector where the names are keys of
\code{obsm} in the new \code{AnnData} object and values are the names of items in the
corresponding slot of \code{x}. See below for details.}

\item{varm_mapping}{A named character vector where the names are keys of
\code{varm} in the new \code{AnnData} object and values are the names of items in the
corresponding slot of \code{x}. See below for details.}

\item{obsp_mapping}{A named character vector where the names are keys of
\code{obsp} in the new \code{AnnData} object and values are the names of items in the
corresponding slot of \code{x}. See below for details.}

\item{varp_mapping}{A named character vector where the names are keys of
\code{varp} in the new \code{AnnData} object and values are the names of items in the
corresponding slot of \code{x}. See below for details.}

\item{uns_mapping}{A named character vector where the names are keys of \code{uns}
in the new \code{AnnData} object and values are the names of items in the
corresponding slot of \code{x}. See below for details.}

\item{assay_name}{For \code{\link[SeuratObject:Seurat-class]{SeuratObject::Seurat}} objects, the name of the assay
to be converted. If \code{NULL}, the default assay will be used
(\code{\link[SeuratObject:DefaultAssay]{SeuratObject::DefaultAssay()}}). This is ignored for other objects.}

\item{output_class}{The \code{AnnData} class to convert to. Must be one of
\code{"HDF5AnnData"} or \code{"InMemoryAnnData"}.}

\item{...}{Additional arguments passed to the generator function for
\code{output_class}}
}
\value{
An \code{AnnData} object of the class requested by \code{output_class}
containing the data specified in the mapping arguments.
}
\description{
Convert other objects to an \code{AnnData} object. See the sections below for
details on how slots are mapped between objects. For more information on the
functionality of an \code{AnnData} object, see \link{AnnData-usage}.
}
\section{Details of mapping arguments}{


All mapping arguments except for \code{x_mapping} expect a named character
vector where names are the keys of the slot in the \code{AnnData} object and
values are the names of items in the corresponding slot of \code{x}. If \code{TRUE},
the conversion function will guess which items to copy as described in the
conversion tables for each object type. In most cases, the default is to
copy all items using the same names except where the correspondence between
objects is unclear. To avoid copying anything to a slot, set the mapping
argument to \code{FALSE}. Empty mapping arguments (\code{NULL}, \code{c()}, \code{list()}) will
be treated as \code{FALSE} with a warning. If an unnamed vector is provided, the
values will be used as names.
\itemize{
\item \code{TRUE} will guess which items to copy as described in the conversion
tables for each object type
\item \code{c(adata_item = "x_item")} will copy \code{x_item} from the slot in \code{x} to
\code{adata_item} in the corresponding slot of new \code{AnnData} object
\item \code{FALSE} will avoid copying anything to the slot
\item \code{c("x_item")} is equivalent to \code{c(x_item = "x_item")}
}
}

\section{Converting from a \code{SingleCellExperiment} object}{


This table describes how slots in a
\code{\link[SingleCellExperiment:SingleCellExperiment]{SingleCellExperiment::SingleCellExperiment}} object to the new \code{AnnData}
object.\tabular{llll}{
   \strong{From \code{SingleCellExperiment}} \tab \strong{To \code{AnnData}} \tab \strong{Example mapping argument} \tab \strong{Default} \cr
   \code{assays(x)} \tab \code{adata$X} \tab \code{x_mapping = "counts"} \tab Nothing is copied to \code{X} \cr
   \code{assays(x)} \tab \code{adata$layers} \tab \code{layers_mapping = c(counts = "counts")} \tab All items are copied by name \cr
   \code{colData(x)} \tab \code{adata$obs} \tab \code{obs_mapping = c(n_counts = "n_counts", cell_type = "CellType")} \tab All columns are copied by name \cr
   \code{rowData(x)} \tab \code{adata$var} \tab \code{var_mapping = c(n_cells = "n_cells", pct_zero = "PctZero")} \tab All columns are copied by name \cr
   \code{reducedDims(x)} \tab \code{adata$obsm} \tab \code{obsm_mapping = c(X_pca = "pca")} \tab All items are copied by name \cr
   \code{featureLoadings(reducedDims(x))} \tab \code{adata$varm} \tab \code{varm_mapping = c(PCs = "pca")} \tab Feature loadings from all \code{\link[SingleCellExperiment:LinearEmbeddingMatrix]{SingleCellExperiment::LinearEmbeddingMatrix}} objects in \code{reducedDims(x)} \cr
   \code{colPairs(x)} \tab \code{adata$obsp} \tab \code{obsp_mapping = c(connectivities = "RNA_nn")} \tab All items are copied by name \cr
   \code{rowPairs(x)} \tab \code{adata$varp} \tab \code{varp_mapping = c(similarities = "gene_overlaps")} \tab All items are copied by name \cr
   \code{metadata(x)} \tab \code{adata$uns} \tab \code{uns_mapping = c(metadata = "project_metadata")} \tab All items are copied by name \cr
}
}

\section{Converting from a \code{Seurat} object}{


Only one assay can be converted from a \code{\link[SeuratObject:Seurat-class]{SeuratObject::Seurat}} object to
an \code{AnnData} object at a time. This can be controlled using the
\code{assay_name} argument. By default, the current default assay will be used.

This table describes how slots in a \code{\link[SeuratObject:Seurat-class]{SeuratObject::Seurat}} object to the
new \code{AnnData} object.\tabular{llll}{
   \strong{From \code{Seurat}} \tab \strong{To \code{AnnData}} \tab \strong{Example mapping argument} \tab \strong{Default} \cr
   \code{Layers(x)} \tab \code{adata$X} \tab \code{x_mapping = "counts"} \tab Nothing is copied to \code{X} \cr
   \code{Layers(x)} \tab \code{adata$layers} \tab \code{layers_mapping = c(counts = "counts")} \tab All items are copied by name \cr
   \code{x[[]]} \tab \code{adata$obs} \tab \code{obs_mapping = c(n_counts = "n_counts", cell_type = "CellType")} \tab All columns are copied by name \cr
   \code{x[[assay_name]][[]]} \tab \code{adata$var} \tab \code{var_mapping = c(n_cells = "n_cells", pct_zero = "PctZero")} \tab All columns are copied by name \cr
   \code{Reductions(x)} \tab \code{adata$obsm} \tab \code{obsm_mapping = c(X_pca = "pca")} \tab All embeddings matching \code{assay_name} are copied by name \cr
   \code{Loadings(x)} \tab \code{adata$varm} \tab \code{varm_mapping = c(PCs = "pca")} \tab All valid loadings are copied by name \cr
   \code{Graphs(x)} \tab \code{adata$obsp} \tab \code{obsp_mapping = c(connectivities = "RNA_nn")} \tab All graphs matching \code{assay_name} are copied by name \cr
   \code{Misc(x)} \tab \code{adata$varp} \tab \code{varp_mapping = c(similarities = "gene_overlaps")} \tab No data is copied to \code{varp} \cr
   \code{Misc(x)} \tab \code{adata$uns} \tab \code{uns_mapping = c(metadata = "project_metadata")} \tab All items are copied by name \cr
}

\subsection{Graph conversion}{

By default, all graphs in a \code{\link[SeuratObject:Seurat-class]{SeuratObject::Seurat}} object that match the
assay being converted are copied to the \code{obsp} slot of the new \code{AnnData}
object. If a graph does not have an associated assay:
\itemize{
\item If \code{assay_name} is the default assay, they will be \emph{converted} with a
warning
\item if \code{assay_name} is not the default assay, they will be \emph{skipped} with a
warning
}

To override this behavior, provide a custom mapping using the
\code{obsp_mapping} argument.
}

\subsection{Unexpected dimensions}{

A \code{\link[SeuratObject:Seurat-class]{SeuratObject::Seurat}} is more flexible in terms of the dimensions of
items that can be stored in various slots. For example, a \code{Layer} does not
have to match the dimensions of the whole object. If an item has
unexpected dimensions, it will be skipped with a warning.
}
}

\examples{
\dontshow{if (rlang::is_installed("Seurat")) withAutoprint(\{ # examplesIf}
# Convert a Seurat object to an AnnData object
library(Seurat)

counts <- matrix(rbinom(20000, 1000, .001), nrow = 100)
obj <- CreateSeuratObject(counts = counts)
obj <- NormalizeData(obj)
obj <- FindVariableFeatures(obj)
obj <- ScaleData(obj)
obj <- RunPCA(obj, npcs = 10L)
obj <- FindNeighbors(obj)
obj <- RunUMAP(obj, dims = 1:10)

as_AnnData(obj)
\dontshow{\}) # examplesIf}
\dontshow{if (rlang::is_installed("SingleCellExperiment")) withAutoprint(\{ # examplesIf}
# Convert a SingleCellExperiment object to an AnnData object
library(SingleCellExperiment)

sce <- SingleCellExperiment(
  assays = list(counts = matrix(1:5, 5L, 3L)),
  colData = DataFrame(cell = 1:3, row.names = paste0("Cell", 1:3)),
  rowData = DataFrame(gene = 1:5, row.names = paste0("Gene", 1:5))
)

as_AnnData(sce)
\dontshow{\}) # examplesIf}
}
\seealso{
Other AnnData creators: 
\code{\link{AnnData}()},
\code{\link{read_h5ad}()}

Other object converters: 
\code{\link{as_HDF5AnnData}()},
\code{\link{as_InMemoryAnnData}()},
\code{\link{as_ReticulateAnnData}()},
\code{\link{as_Seurat}()},
\code{\link{as_SingleCellExperiment}()},
\code{\link{reticulate-helpers}}
}
\concept{AnnData creators}
\concept{object converters}
