#' Average TSA Curves
#'
#' This function will take either Fluorescence or Normalized Fluorescence
#'     curves from the submitted data frame and find the average (mean) and
#'     standard deviation (sd) for each temperature measured in the TSA curve.
#'     Mean and sd are smoothened by default to generate cleaner curves.
#'     The function \code{\link[mgcv]{gam}} from the mgcv package is used for
#'     regression to smoothen lines. Smoothing can be turned off and the true
#'     average for each point can be given, however, plots will look messier.
#'     The qPCR machine may return temperatures with many decimal places,
#'     and TSAR only merges identical values, therefore rounding is necessary.
#'     Data is rounded to one decimal place to improve regression smoothing.
#'     \cr \cr
#'     \strong{Note:} All submitted data is averaged, regardless of condition or
#'     well ID.
#'     If you wish to average by condition, you will need to sort the data frame
#'     and run this function on subsets.
#'
#' @importFrom magrittr %>%
#' @importFrom dplyr summarize group_by n_distinct
#'
#' @param tsa_data a data frame that is merged and generated by
#'     TSAR::merge_TSA(). If \code{y = 'RFU'}, tsa_data must also be
#'     generated by TSAR::normalize_fluorescence. The Temperature column will be
#'     rounded and the average & sd of each rounded temperature is calculated.
#' @param y character string; c('Fluorescence', 'RFU').
#'     When \code{y = 'Fluorescence'}, the original Fluorescence data from
#'     TSAR::read_raw_data() is averaged. When \code{y = 'RFU'}, the average is
#'     calculated by the rescaled fluorescence.
#' @param digits an integer; \code{digits = 1} by default.
#'     The number of decimal places to round temperature to for averaging.
#' @param avg_smooth,sd_smooth logical; TRUE by default. Decides if the
#'    average (avg_smooth) or standard deviation (sd_smooth) will be smoothened
#'    by regression via mgcv::gam()
#'
#' @return a data frame of each temperature measured with the average, sd, and
#'    n(# of averaged values) calculated. Depending on avg_smooth and sd_smooth,
#'    the smoothened lines for the maximum and mimimum sd and the average will
#'    also be returned.
#' @family TSAR Formatting
#' @seealso \code{\link{merge_TSA}} and \code{\link{merge_TSA}} for
#'    preparing data.
#'
#' @examples
#' data("example_tsar_data")
#' TSA_average(example_tsar_data,
#'     y = "Fluorescence", digits = 1,
#'     avg_smooth = TRUE, sd_smooth = TRUE
#' )
#'
#' @export
TSA_average <- function(tsa_data,
                        y = "Fluorescence",
                        digits = 1,
                        avg_smooth = TRUE,
                        sd_smooth = TRUE) {

    y <- match.arg(y, choices = c("Fluorescence", "RFU"))
    tsa_data_new <- tsa_data
    tsa_data_new$Temperature <- round(tsa_data_new$Temperature, digits = digits)
    if (y == "Fluorescence") {
        tsa_data_new <- tsa_data_new %>%
            group_by(Temperature) %>%
            summarize(
                average = mean(Fluorescence),
                n = n_distinct(Fluorescence),
                sd = sd(Fluorescence)
            )
    }
    if (y == "RFU") {
        tsa_data_new <- tsa_data_new %>%
            group_by(Temperature) %>%
            summarize(
                average = mean(Normalized),
                n = n_distinct(Normalized),
                sd = sd(Normalized)
            )
    }
    tsa_data_new$sd_max <- tsa_data_new$average + tsa_data_new$sd
    tsa_data_new$sd_min <- tsa_data_new$average - tsa_data_new$sd
    if (avg_smooth) {
        df <- tsa_data_new[, c("Temperature", "average")]
        names(df) <- c("x", "y")
        m <- gam(y ~ s(x), data = df)
        p <- predict(m, newdata = data.frame(x = df$x))
        tsa_data_new$avg_smooth <- p
    }
    if (sd_smooth) {
        df <- tsa_data_new[, c("Temperature", "sd_min")]
        names(df) <- c("x", "y")
        m <- gam(y ~ s(x), data = df)
        p <- predict(m, newdata = data.frame(x = df$x))
        tsa_data_new$sd_min_smooth <- p

        df <- tsa_data_new[, c("Temperature", "sd_max")]
        names(df) <- c("x", "y")
        m <- gam(y ~ s(x), data = df)
        p <- predict(m, newdata = data.frame(x = df$x))
        tsa_data_new$sd_max_smooth <- p
    }
    return(tsa_data_new)
}
