% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MakeSE.R
\name{makeSE}
\alias{makeSE}
\title{Imports a collated dataset into the R session as an NxtSE object}
\usage{
makeSE(
  collate_path,
  colData,
  RemoveOverlapping = TRUE,
  realize = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{collate_path}{(Required) The output path of \link{collateData} pointing
to the collated data}

\item{colData}{(Optional) A data frame containing the sample annotation
information. The first column must contain the sample names.
Omit \code{colData} to generate a NxtSE object of the whole dataset without
any assigned annotations.
Alternatively, if the names of only a subset of samples are given, then
\code{makeSE()} will construct the NxtSE object based only on the samples given.
The colData can be set later using \link{colData}}

\item{RemoveOverlapping}{(default = \code{TRUE}) Whether to filter out overlapping
IR events belonging to minor isoforms. See details.}

\item{realize}{(default = \code{FALSE}) Whether to load all assay data into
memory. See details}

\item{verbose}{(default = \code{TRUE}) Whether loading messages are displayed}
}
\value{
A \linkS4class{NxtSE} object containing the compiled data in
DelayedArrays (or as matrices if \code{realize = TRUE}), pointing to the assay
data contained in the given \code{collate_path}
}
\description{
Creates a \linkS4class{NxtSE} object from the data (that was collated using
\link{collateData}). This object is used
for downstream differential analysis of IR and alternative splicing events
using \link{ASE-methods}, data generation for visualization of scatter plots and
heatmaps via \link{make_plot_data} methods, and coverage visualisation using
\link{plotCoverage}
}
\details{
\code{makeSE} retrieves the data collated by
\link{collateData}, and initialises a \linkS4class{NxtSE} object. It references
the required on-disk assay data using DelayedArrays, thereby utilising
'on-disk' memory to conserve memory usage.

For extremely large datasets, loading the entire data into memory may consume
too much memory. In such cases, make a subset of the \linkS4class{NxtSE}
object (e.g. subset by samples) before loading the data into memory (RAM)
using \link{realize_NxtSE}. Alternatively supply a data frame to the \code{colData}
parameter of the \code{makeSE()} function. Only samples listed in the first column
of the \code{colData} data frame will be imported into the \linkS4class{NxtSE}
object.

It should be noted that downstream applications of SpliceWiz, including
\link{ASE-methods}, \link{plotCoverage}, are much faster if the \linkS4class{NxtSE}
is realized. It is recommended to realize the \linkS4class{NxtSE} object
before extensive usage.

If COV files assigned via \link{collateData} have been moved relative to the
\code{collate_path}, the created \linkS4class{NxtSE} object will not be linked to
any COV files and \link{plotCoverage} cannot be used. To reassign these
files, a vector of file paths corresponding to all the COV files of the data
set can be assigned using \code{covfile(se) <- vector_of_cov_files}. See
the example below for details.

If \code{RemoveOverlapping = TRUE}, \code{makeSE} will remove introns that overlap
other introns with higher junction read counts in the dataset. This means
that SpliceWiz will assess a set of non-overlapping introns which belong
to likely major isoforms, ensuring that overlapping
IR events are not 'double-counted'.

NB: Since version 1.3.4, SpliceWiz has improved the algorithm of generating
the set of non-overlapping introns (prior versions appear to generate
sets of introns that still overlap). To use the prior algorithm for
compatibility with prior analysis, set \code{RemoveOverlapping = FALSE}.
}
\examples{

# The following code can be used to reproduce the NxtSE object
# that can be fetched with SpliceWiz_example_NxtSE()

buildRef(
    reference_path = file.path(tempdir(), "Reference"),
    fasta = chrZ_genome(),
    gtf = chrZ_gtf()
)

bams <- SpliceWiz_example_bams()
processBAM(bams$path, bams$sample,
  reference_path = file.path(tempdir(), "Reference"),
  output_path = file.path(tempdir(), "SpliceWiz_Output")
)

expr <- findSpliceWizOutput(file.path(tempdir(), "SpliceWiz_Output"))
collateData(expr,
  reference_path = file.path(tempdir(), "Reference"),
  output_path = file.path(tempdir(), "Collated_output")
)

se <- makeSE(collate_path = file.path(tempdir(), "Collated_output"))

# "Realize" NxtSE object to load all H5 assays into memory:

se <- realize_NxtSE(se)

# If COV files have been removed since the last call to collateData()
# reassign them to the NxtSE object, for example:

covfile_path <- system.file("extdata", package = "SpliceWiz")
covfile_df <- findSamples(covfile_path, ".cov")

covfile(se) <- covfile_df$path

}
