% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normalize.R
\name{msNormalize}
\alias{msNormalize}
\title{Function for performing normalization and batch corrections on imputed data.}
\usage{
msNormalize(
  data,
  normalizeMethod = c("median", "ComBat", "quantile", "quantile + ComBat",
    "median + ComBat", "CRMN", "RUV", "SVA"),
  nControl = 10,
  controls = NULL,
  nComp = 2,
  kRUV = 3,
  batch = "batch",
  covariatesOfInterest = NULL,
  transform = c("log10", "log2", "ln", "none"),
  compVars = c("mz", "rt"),
  sampleVars = c("subject_id"),
  colExtraText = NULL,
  separator = NULL,
  returnToSE = FALSE,
  returnToDF = FALSE
)
}
\arguments{
\item{data}{Data set as either a data frame or `SummarizedExperiement`.}

\item{normalizeMethod}{Name of normalization method.
"ComBat" (only ComBat batch correction), "quantile" (only quantile 
normalization), "quantile + ComBat" (quantile with ComBat batch correction),
"median" (only median normalization), "median + ComBat" (median with ComBat
batch correction), "CRMN" (cross-contribution compensating multiple 
standard normalization), "RUV" (remove unwanted variation), "SVA" (surrogate 
variable analysis)}

\item{nControl}{Number of controls to estimate/utilize (for CRMN and RUV).}

\item{controls}{Vector of control identifiers.  Leave blank for data driven
controls. Vector of column numbers from metafin dataset of that control (for
CRMN and RUV).}

\item{nComp}{Number of factors to use in CRMN algorithm.}

\item{kRUV}{Number of factors to use in RUV algorithm.}

\item{batch}{Name of the sample variable identifying batch.}

\item{covariatesOfInterest}{Sample variables used as covariates in
normalization algorithms (required for ComBat, CRMN, and SVA).}

\item{transform}{Select transformation to apply to data prior to 
normalization. Options are "log10", "log2", "ln" and "none".}

\item{compVars}{Vector of the columns which identify compounds. If a 
`SummarizedExperiment` is used for `data`, row variables will be used.}

\item{sampleVars}{Vector of the ordered sample variables found in each sample
column.}

\item{colExtraText}{Any extra text to ignore at the beginning of the sample 
columns names. Unused for `SummarizedExperiments`.}

\item{separator}{Character or text separating each sample variable in sample
columns. Unused for `SummarizedExperiment`.}

\item{returnToSE}{Logical value indicating whether to return as 
`SummarizedExperiment`}

\item{returnToDF}{Logical value indicating whether to return as data frame.}
}
\value{
A data frame or `SummarizedExperiment` with transformed and
normalized data. Default return type is set to match the data input but may 
be altered with the `returnToSE` or `returnToDF` arguments.
}
\description{
Perform normalization and batch corrections on specified imputed dataset.
Routines included are quantile, RUV (remove unwanted variation), SVA 
(surrogate variable analysis), median, CRMN (cross-contribution 
compensating multiple standard normalization), and ComBat to remove batch 
effects in raw, quantile, and median normalized data. Generates data 
driven controls if none exist.
}
\examples{
# Load, tidy, summarize, filter, and impute example dataset
data(msquant)

summarizedDF <- msSummarize(msquant,
                            compVars = c("mz", "rt"),
                            sampleVars = c("spike", "batch", "replicate", 
                            "subject_id"),
                            cvMax = 0.50,
                            minPropPresent = 1/3,
                            colExtraText = "Neutral_Operator_Dif_Pos_",
                            separator = "_",
                            missingValue = 1)
                            
filteredDF <- msFilter(summarizedDF,
                       filterPercent = 0.8,
                       compVars = c("mz", "rt"),
                       sampleVars = c("spike", "batch", "subject_id"),
                       separator = "_")

hmImputedDF <- msImpute(filteredDF, imputeMethod = "halfmin",
                        compVars = c("mz", "rt"),
                        sampleVars = c("spike", "batch", "subject_id"),
                        separator = "_",
                        missingValue = 0)

# Normalize data set
medianNormalizedDF <- msNormalize(hmImputedDF, normalizeMethod = "median",
                                  compVars = c("mz", "rt"),
                                  sampleVars = c("spike", "batch", 
                                  "subject_id"),
                                  separator = "_")

}
\references{
Bolstad, B.M.et al.(2003) A comparison of normalization methods for high
density oligonucleotide array data based on variance and bias.
Bioinformatics, 19, 185-193

DeLivera, A.M.et al.(2012) Normalizing and Integrating Metabolomic Data.
Anal. Chem, 84, 10768-10776.

Gagnon-Bartsh, J.A.et al.(2012) Using control genes to correct for unwanted
variation in microarray data. Biostatistics, 13, 539-552.

Johnson, W.E.et al.(2007) Adjusting batch effects in microarray expression
data using Empirical Bayes methods. Biostatistics, 8, 118-127.

Leek, J.T.et al.(2007) Capturing Heterogeneity in Gene Expression Studies by
Surrogate Variable Analysis. PLoS Genetics, 3(9), e161

Wang, W.et al.(2003) Quantification of Proteins and Metabolites by Mass
Spectrometry without Isotopic Labeling or Spiked Standards. Anal. Chem., 75,
4818-4826.
}
