#' @title `CoolFile` S4 class
#' 
#' @name CoolFile-class
#' @aliases McoolFile-class
#' @aliases CoolFile
#' 
#' @description
#' 
#' The `CoolFile` class describes a `BiocFile` object, pointing to the location 
#' of an Hi-C matrix file (cool, mcool, hic, hicpro, ...) and containing additional slots:
#' 
#' 1. resolution: at which resolution the associated mcool file should be parsed 
#' 2. pairsFile: the path (in plain character) to an optional pairs file 
#'   (stored as a `PairsFile` object);
#' 3. metadata: a list. If the CoolFile is created by `HiCool`, it will contain 
#'   two elements: `log` (path to `HiCool` processing log file) and `stats` 
#'   (aggregating some stats from `HiCool` mapping).
#' 
#' @param path String; path to a (m)cool file
#' @param resolution numeric; resolution to use with mcool file
#' @param pairsFile String; path to a pairs file
#' @param metadata list; if the CoolFile object was generated by HiCool::HiCool,
#'    this list contains the path to log file, some statistics regarding 
#'    the number of pairs obtained by hicstuff as well as the arguments and 
#'    the hash ID used by HiCool.
#' 
#' @importFrom S4Vectors metadata
#' @importFrom methods setClass
#' @importClassesFrom BiocIO BiocFile
#' @include HiCExperiment-class.R
#' @include PairsFile-class.R
#' @seealso [HicFile()], [HicproFile()]
#' 
#' @examples
#' mcoolPath <- HiContactsData::HiContactsData('yeast_wt', 'mcool')
#' pairsPath <- HiContactsData::HiContactsData('yeast_wt', 'pairs.gz')
#' cf <- CoolFile(
#'   mcoolPath, 
#'   resolution = 2000, 
#'   pairsFile = pairsPath, 
#'   metadata = list(info = 'Yeast WT Hi-C exp.')
#' )
#' cf
#' resolution(cf)
#' pairsFile(cf)
#' metadata(cf)
NULL

#' @export

setClass('CoolFile', contains = 'ContactsFile')

#' @export

setClass('McoolFile', contains = 'CoolFile')

#' @export 

CoolFile <- function(path, resolution = NULL, pairsFile = NULL, metadata = list()) {
    path <- gsub('~', Sys.getenv('HOME'), path)
    .check_cool_file(path)
    if (.is_mcool(path) & is.null(resolution)) 
        resolution <- .lsCoolResolutions(path)[1]
    .check_cool_format(path, resolution)
    new(
        'CoolFile', 
        resource = path, 
        resolution = resolution,
        pairsFile = PairsFile(pairsFile),
        metadata = metadata
    )
}
