### ============================================================================
### sangerReadF ABIF Initial test
### ============================================================================
test_that("sangerReadF Initial test", {
    expect_type(sangerReadF, "S4")
    expect_s4_class(sangerReadF, "SangerRead")
    expect_equal(sangerReadF@inputSource, "ABIF")
    expect_equal(sangerReadF@readFeature, "Forward Read")
    expect_equal(basename(sangerReadF@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(sangerReadF@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(sangerReadF@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(sangerReadF@primaryAASeqS1), "LGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*R")
    expect_equal(as.character(sangerReadF@primaryAASeqS2), "WASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEE")
    expect_equal(as.character(sangerReadF@primaryAASeqS3), "GRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKR")
    expect_equal(sangerReadF@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(sangerReadF@ChromatogramParam@heightPerRow, 200)
    expect_equal(sangerReadF@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(sangerReadF@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(sangerReadF@QualityReport@TrimmingMethod, "M1")
    expect_equal(sangerReadF@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(sangerReadF@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(sangerReadF@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(sangerReadF@QualityReport@qualityPhredScores), 702)
    expect_equal(length(sangerReadF@QualityReport@qualityBaseScores), 702)

    expect_equal(sangerReadF@QualityReport@rawSeqLength, 702)
    expect_equal(sangerReadF@QualityReport@trimmedSeqLength, 461)
    expect_equal(sangerReadF@QualityReport@trimmedStartPos, 16)
    expect_equal(sangerReadF@QualityReport@trimmedFinishPos, 477)
    expect_equal(sangerReadF@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(sangerReadF@QualityReport@trimmedMeanQualityScore, 58.21041, tolerance=1e-6)
    expect_equal(sangerReadF@QualityReport@rawMinQualityScore, 1)
    expect_equal(sangerReadF@QualityReport@trimmedMinQualityScore, 13)
    expect_equal(sangerReadF@QualityReport@remainingRatio, 0.6566952, tolerance=1e-6)
})

### ============================================================================
### sangerReadF ABIF Make Base Call test
### ============================================================================
test_that("sangerReadF Make Base Call 1 (normal value)", {
    baseCall <- MakeBaseCalls(sangerReadF, signalRatioCutoff = 0.23)
    expect_type(baseCall, "S4")
    expect_s4_class(baseCall, "SangerRead")
    expect_equal(baseCall@readFeature, "Forward Read")
    expect_equal(basename(baseCall@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(baseCall@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(baseCall@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGACGTTCAGTCATAGTGAATTCCAGA")
    expect_equal(as.character(baseCall@primaryAASeqS1), "LGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*R")
    expect_equal(as.character(baseCall@primaryAASeqS2), "WASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEE")
    expect_equal(as.character(baseCall@primaryAASeqS3), "GRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKR")
    expect_equal(baseCall@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(baseCall@ChromatogramParam@heightPerRow, 200)
    expect_equal(baseCall@ChromatogramParam@signalRatioCutoff, 0.23)
    expect_equal(baseCall@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(baseCall@QualityReport@TrimmingMethod, "M1")
    expect_equal(baseCall@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(baseCall@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(baseCall@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(baseCall@QualityReport@qualityPhredScores), 702)
    expect_equal(length(baseCall@QualityReport@qualityBaseScores), 702)

    expect_equal(baseCall@QualityReport@rawSeqLength, 702)
    expect_equal(baseCall@QualityReport@trimmedSeqLength, 461)
    expect_equal(baseCall@QualityReport@trimmedStartPos, 16)
    expect_equal(baseCall@QualityReport@trimmedFinishPos, 477)
    expect_equal(baseCall@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(baseCall@QualityReport@trimmedMeanQualityScore, 58.21041, tolerance=1e-6)
    expect_equal(baseCall@QualityReport@rawMinQualityScore, 1)
    expect_equal(baseCall@QualityReport@trimmedMinQualityScore, 13)
    expect_equal(baseCall@QualityReport@remainingRatio, 0.6566952, tolerance=1e-6)
})
test_that("sangerReadF Make Base Call 2 (small value)", {
    baseCall2 <- MakeBaseCalls(sangerReadF, signalRatioCutoff = 0.01)
    expect_type(baseCall2, "S4")
    expect_s4_class(baseCall2, "SangerRead")
    expect_equal(baseCall2@readFeature, "Forward Read")
    expect_equal(basename(baseCall2@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(baseCall2@primarySeq), "CACTTTATANTNTATTCTNGGCGTCNGAGCAGGAATGGTTGGAGCNNGTATAAGANTTCTANTNCGAATCGAGCNAAGACANCNANGANCNTNCNNGGGCAGAGACCAANTNTANNNNNCNNNCNNNNNNGNNNNCNCNTTTGTAATAATCTTCTTTCTAGTAATGCCNGNATNNNNCGGGGGANNNNGAANCNGNCTTTTANNNTTANNANNNGGAGNCCNCNANNNNNNNTTCCCTCNACTCNANANNANNANANNCNGANTACTTCNCCCNTCNCNGANNCNTNTANTGTCCNCTGCGGCGNTAGAANAAGGCGCNGGNANGGNGTGAACNGTTTNTCNGCCTCTAGCANNAANNCNNGCNNACGNNGGCNCGNCTGTAGATTTANCCNTCTTTTCCCTTNATTTANNGGGTGCNTCNNCNATTCTAGGGNCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATNACGTCTNGAACGAATTCCNNNGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTNCTACTTCTATCTNTACCAGTGCTANCAGGTGCCANTACCATACTTCTTACNGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATNNNNAGATCACCCNGATGTNGAGTCATANTGAANTCCTGA")
    expect_equal(as.character(baseCall2@secondarySeq), "KYGTMCSSANMNSSGSCRNYAWATGNMTGWKYGTGGWTTRTWYGANNWASGCKWTNAMWMCNRNKMTKMKMSMWNCGACGKNKNSNAKNKNRNKNNAAMAKAAMCGACSNMNMYNNNNNWNNNKNNNNNNMNNNNTNRNAARWRKKMYKCTMMTMAMTMSWRGKRMGKNMNKRNNNNKMAAAAKNNNNWTKNWNWNKAMAAYNNNRRKNNSNNNMMSMNKWNKNYNNNNNNNMCTTWMRNCKMRNSNKNNKNNKNTNNKNMCNMSWMCTNWTWNMWNTNWKNNKNSNMSNSCSGKNKRMRAARANRSMGSNKKMMGMKNAWNSNMANWRWKYTNYSMMNMTNMWWMWMSMWSNNCSNNTNNMTNNSKMNNYMCNKWNTRAMGMTTMASNTWNCTMMMMTWTAMNKCRRSNNAMWMMTNMGNNKNGGSCAAGGTNTAGSMCYCCCMCSCGTWKTAWMMSGRGKGGRCTWARAAKGATWKSNGTWMCANMCSKMCKTGKKNNNAARRMMGRAAATAAMMWGGAGAAKAAMMAAMCRNCAGWMACGSSGRNSSGRAYAGTGCNTKYWMATTGNASTKGGGCRAGATGTNMAWTGCGTTCAASKTGGAAGGTGGARAGAGCGAATTAAAMMAMCYGARRGGSGAGACAGKGGGKGRRTCCAGRCGCNNNNCMAACTGKWNCCCMANCCGAGGGANAAAANGRCAGG")
    expect_equal(as.character(baseCall2@primaryAASeqS1), "LGVXAGMVGAXIRXLXRIEXRXXXXXXGRDQXXXXXXXXXXFVIIFFLVMPXXXGGXXXXLLXLXXGXXXXXFPXLXXXXXXXLXPSXXXXXSXAAXEXGAGXX*TVXXPLAXXXAXXGXXVDLXXFSLXLXGASXILGXINFITTVINMR*R")
    expect_equal(as.character(baseCall2@primaryAASeqS2), "XASEQEWLEXV*XFXXESSXDXXXXXXAETXXXXXXXXXXXL**SSF**CXXXXGXXEXXFXXXXXEXXXXXSLXSXXXXXXYFXXXXXXXCPLRR*XKAXXXXEXFXXL*XXXXXTXARL*IXPSFPXIXXVXXXF*GLLILSPQLLICVEE")
    expect_equal(as.character(baseCall2@primaryAASeqS3), "XRXSRNGWSXYKXSXXNRAKTXXXXXGQRPXXXXXXXXXXXCNNLLSSNAXXXRGXXXXXFXXXXXXPXXXXPSTXXXXXXXTSPXXXXXXVXCGXRXRRXXGVNXXSASSXXXXXRXXXCRFXXLFPXFXGCXXXSRXY*FYHHSY*YALKR")
    expect_equal(baseCall2@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(baseCall2@ChromatogramParam@heightPerRow, 200)
    expect_equal(baseCall2@ChromatogramParam@signalRatioCutoff, 0.01)
    expect_equal(baseCall2@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(baseCall2@QualityReport@TrimmingMethod, "M1")
    expect_equal(baseCall2@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(baseCall2@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(baseCall2@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(baseCall2@QualityReport@qualityPhredScores), 702)
    expect_equal(length(baseCall2@QualityReport@qualityBaseScores), 702)

    expect_equal(baseCall2@QualityReport@rawSeqLength, 702)
    expect_equal(baseCall2@QualityReport@trimmedSeqLength, 461)
    expect_equal(baseCall2@QualityReport@trimmedStartPos, 16)
    expect_equal(baseCall2@QualityReport@trimmedFinishPos, 477)
    expect_equal(baseCall2@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(baseCall2@QualityReport@trimmedMeanQualityScore, 58.21041, tolerance=1e-6)
    expect_equal(baseCall2@QualityReport@rawMinQualityScore, 1)
    expect_equal(baseCall2@QualityReport@trimmedMinQualityScore, 13)
    expect_equal(baseCall2@QualityReport@remainingRatio, 0.6566952, tolerance=1e-6)
})
test_that("sangerReadF Make Base Call 3 (equal to 1)", {
    baseCall3 <- MakeBaseCalls(sangerReadF, signalRatioCutoff = 1)
    expect_type(baseCall3, "S4")
    expect_s4_class(baseCall3, "SangerRead")
    expect_equal(baseCall3@readFeature, "Forward Read")
    expect_equal(basename(baseCall3@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(baseCall3@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(baseCall3@secondarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(baseCall3@primaryAASeqS1), "LGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*R")
    expect_equal(as.character(baseCall3@primaryAASeqS2), "WASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEE")
    expect_equal(as.character(baseCall3@primaryAASeqS3), "GRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKR")
    expect_equal(baseCall3@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(baseCall3@ChromatogramParam@heightPerRow, 200)
    expect_equal(baseCall3@ChromatogramParam@signalRatioCutoff, 1)
    expect_equal(baseCall3@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(baseCall3@QualityReport@TrimmingMethod, "M1")
    expect_equal(baseCall3@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(baseCall3@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(baseCall3@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(baseCall3@QualityReport@qualityPhredScores), 702)
    expect_equal(length(baseCall3@QualityReport@qualityBaseScores), 702)

    expect_equal(baseCall3@QualityReport@rawSeqLength, 702)
    expect_equal(baseCall3@QualityReport@trimmedSeqLength, 461)
    expect_equal(baseCall3@QualityReport@trimmedStartPos, 16)
    expect_equal(baseCall3@QualityReport@trimmedFinishPos, 477)
    expect_equal(baseCall3@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(baseCall3@QualityReport@trimmedMeanQualityScore, 58.21041, tolerance=1e-6)
    expect_equal(baseCall3@QualityReport@rawMinQualityScore, 1)
    expect_equal(baseCall3@QualityReport@trimmedMinQualityScore, 13)
    expect_equal(baseCall3@QualityReport@remainingRatio, 0.6566952, tolerance=1e-6)
})
# test_that("sangerReadF Make Base Call 4 (bigger than 1)", {
#     expect_error(MakeBaseCalls(sangerReadF, signalRatioCutoff = 1.2),
#                  "'signalRatioCutoff' must be between 0 and 1.", fixed = TRUE)
# })
test_that("sangerReadF Make Base Call 5 (equal to 0)", {
    baseCall5 <- MakeBaseCalls(sangerReadF, signalRatioCutoff = 0)
    expect_type(baseCall5, "S4")
    expect_s4_class(baseCall5, "SangerRead")
    expect_equal(baseCall5@readFeature, "Forward Read")
    expect_equal(basename(baseCall5@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(baseCall5@primarySeq), "NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN")
    expect_equal(as.character(baseCall5@secondarySeq), "NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN")
    expect_equal(as.character(baseCall5@primaryAASeqS1), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX")
    expect_equal(as.character(baseCall5@primaryAASeqS2), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX")
    expect_equal(as.character(baseCall5@primaryAASeqS3), "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX")
    expect_equal(baseCall5@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(baseCall5@ChromatogramParam@heightPerRow, 200)
    expect_equal(baseCall5@ChromatogramParam@signalRatioCutoff, 0)
    expect_equal(baseCall5@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(baseCall5@QualityReport@TrimmingMethod, "M1")
    expect_equal(baseCall5@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(baseCall5@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(baseCall5@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(baseCall5@QualityReport@qualityPhredScores), 702)
    expect_equal(length(baseCall5@QualityReport@qualityBaseScores), 702)

    expect_equal(baseCall5@QualityReport@rawSeqLength, 702)
    expect_equal(baseCall5@QualityReport@trimmedSeqLength, 461)
    expect_equal(baseCall5@QualityReport@trimmedStartPos, 16)
    expect_equal(baseCall5@QualityReport@trimmedFinishPos, 477)
    expect_equal(baseCall5@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(baseCall5@QualityReport@trimmedMeanQualityScore, 58.21041, tolerance=1e-6)
    expect_equal(baseCall5@QualityReport@rawMinQualityScore, 1)
    expect_equal(baseCall5@QualityReport@trimmedMinQualityScore, 13)
    expect_equal(baseCall5@QualityReport@remainingRatio, 0.6566952, tolerance=1e-6)
})
# test_that("sangerReadF Make Base Call 6 (smaller than 0)", {
#     expect_error(MakeBaseCalls(sangerReadF, signalRatioCutoff = -0.01),
#                  "'signalRatioCutoff' must be between 0 and 1.", fixed = TRUE)
# })
test_that("sangerReadR Make Base Call 7 (normal value) - reverse", {
    baseCallR <- MakeBaseCalls(sangerReadR, signalRatioCutoff = 0.23)
    expect_type(baseCallR, "S4")
    expect_s4_class(baseCallR, "SangerRead")
    expect_equal(baseCallR@readFeature, "Reverse Read")
    expect_equal(basename(baseCallR@readFileName), "Achl_ACHLO006-09_2_R.ab1")
    expect_equal(as.character(baseCallR@primarySeq), "CCGTAGTAGGTGTTGGTAGAGGATGGGGTCTCCACCACCGGCAGGATCAAAGAATGAAGTATTGAGGTTTCGGTCGGTAAGAAGTATGGTAATGGCACCTGCTAGCACTGGTAAAGATAGAAGTAGAAGAACAACTGTAATTAGCACAGCTCAGACAAACAGGGGAATTCGTTCAAGACGTAATCCTCTTCAACGCATATTAATAACTGTGGTGATAAAATTAATAGCCCCTAGAATAGAAGACGCACCCGCTAAATGAAGGGAAAAGATGGCTAAATCTACAGACGGGCCTGCGTGGGCAAGATTTCTTGCTAGAGGCGGATAAACAGTTCACCCCGTACCAGCGCCTTTTTCTACCGCCGCAGAGGACACTAAAAGGATCAGTGATGGGGGAAGTAGTCAGAATCTCATGTTGTTGAGTCGAGGGAATGCTATATCGGGGGCTCCAAGTATTAAAGGTAAAAGCCAGTTTCCGAATCCCCCGATGAATACAGGCATTACTAGAAAGAAGATTATTACAAATGCGTGTGCAGTAACGATAGTATTGTATAGTTGGTCTCTGCCCAGGAACGCTCCTGGTTGTCTTAGCTCGATTCGAATTAGAAGTCTTATACCGGCTCCAACCATTCCTGCTCAGACGCCCAGAATAAAATATAAAGTTCCAATATCTTTATGATTTGTTGACCACTGGCCGTCGATATACAA")
    expect_equal(as.character(baseCallR@secondarySeq), "CAGCAGTATGTGTTGGTAGAGGATGGGGTCTCCACCACCGGCAGGATCAAAGAATGAAGTATTGCGGTTTCGGTCGGTAAGAAGTATGGTAATGGCACCTGCTAGCACTGGTAAAGATAGAAGTAGAAGAACAACTGTAATTAGCACAGCTCAGACAAACAGGGGAATTCGTTCAAGACGTAATCCTCTTCAACGCATATTAATAACTGTGGTGATAAAATTAATAGCCCCTAGAATAGAAGACGCACCCGCTAAATGAAGGGAAAAGATGGCTAAATCTACAGACGGGCCTGCGTGGGCAAGATTTCTTGCTAGAGGCGGATAAACAGTTCACCCCGTACCAGCGCCTTTTTCTACCGCCGCAGAGGACACTAAAAGGATCAGTGATGGGGGAAGTAGTCAGAATCTCATGTTGTTGAGTCGAGGGAATGCTATATCGGGGGCTCCAAGTATTAAAGGTAAAAGCCAGTTTCCGAATCCCCCGATGAATACAGGCATTACTAGAAAGAAGATTATTACAAATGCGTGTGCAGTAACGATAGTATTGTATAGTTGGTCTCTGCCCAGGAACGCTCCTGGTTGTCTTAGCTCGATTCGAATTAGAAGTCTTATACCGGCTCCAACCATTCCTGCTCAGACGCCCAAAATAAAATATAAAGTTCCAATATCTTTATGATTTGTTGACCACTGGCCGTCGATGTACAA")
    expect_equal(as.character(baseCallR@primaryAASeqS1), "EDGVSTTGRIKE*SIEVSVGKKYGNGTC*HW*R*K*KNNCN*HSSDKQGNSFKT*SSSTHINNCGDKINSP*NRRRTR*MKGKDG*IYRRACVGKISC*RRINSSPRTSAFFYRRRGH*KDQ*WGK*SESHVVESRECYIGGSKY*R*KPVSESPDEYRHY*KEDYYKCVCSNDSIV*LVSAQERSWLS*LDSN*KSYTGSNHSCSDAQNKI*")
    expect_equal(as.character(baseCallR@primaryAASeqS2), "RMGSPPPAGSKNEVLRFRSVRSMVMAPASTGKDRSRRTTVISTAQTNRGIRSRRNPLQRILITVVIKLIAPRIEDAPAK*REKMAKSTDGPAWARFLARGG*TVHPVPAPFSTAAEDTKRISDGGSSQNLMLLSRGNAISGAPSIKGKSQFPNPPMNTGITRKKIITNACAVTIVLYSWSLPRNAPGCLSSIRIRSLIPAPTIPAQTPRIKYK")
    expect_equal(as.character(baseCallR@primaryAASeqS3), "GWGLHHRQDQRMKY*GFGR*EVW*WHLLALVKIEVEEQL*LAQLRQTGEFVQDVILFNAY**LW**N**PLE*KTHPLNEGKRWLNLQTGLRGQDFLLEADKQFTPYQRLFLPPQRTLKGSVMGEVVRISCC*VEGMLYRGLQVLKVKASFRIPR*IQALLERRLLQMRVQ*R*YCIVGLCPGTLLVVLARFELEVLYRLQPFLLRRPE*NI")
    expect_equal(baseCallR@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(baseCallR@ChromatogramParam@heightPerRow, 200)
    expect_equal(baseCallR@ChromatogramParam@signalRatioCutoff, 0.23)
    expect_equal(baseCallR@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(baseCallR@QualityReport@TrimmingMethod, "M1")
    expect_equal(baseCallR@QualityReport@M1TrimmingCutoff, 0.0001, tolerance=1e-6)
    expect_equal(baseCallR@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(baseCallR@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(baseCallR@QualityReport@qualityPhredScores), 705)
    expect_equal(length(baseCallR@QualityReport@qualityBaseScores), 705)

    expect_equal(baseCallR@QualityReport@rawSeqLength, 705)
    expect_equal(baseCallR@QualityReport@trimmedSeqLength, 640)
    expect_equal(baseCallR@QualityReport@trimmedStartPos, 18)
    expect_equal(baseCallR@QualityReport@trimmedFinishPos, 658)
    expect_equal(baseCallR@QualityReport@rawMeanQualityScore, 54.85106, tolerance=1e-6)
    expect_equal(baseCallR@QualityReport@trimmedMeanQualityScore, 57.29531, tolerance=1e-6)
    expect_equal(baseCallR@QualityReport@rawMinQualityScore, 3)
    expect_equal(baseCallR@QualityReport@trimmedMinQualityScore, 6)
    expect_equal(baseCallR@QualityReport@remainingRatio, 0.9078014, tolerance=1e-6)
})

### ============================================================================
### sangerReadF ABIF Update Quality Trimming
### ============================================================================
test_that("sangerReadF update quality trimming parameters 1 (smaller M1)", {
    trimmedsangerReadF <- updateQualityParam(sangerReadF,
                                            TrimmingMethod       = "M1",
                                            M1TrimmingCutoff     = 0.000001,
                                            M2CutoffQualityScore = NULL,
                                            M2SlidingWindowSize  = NULL)
    expect_type(trimmedsangerReadF, "S4")
    expect_s4_class(trimmedsangerReadF, "SangerRead")
    expect_equal(trimmedsangerReadF@inputSource, "ABIF")
    expect_equal(trimmedsangerReadF@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF@primaryAASeqS1), "MVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIF")
    expect_equal(as.character(trimmedsangerReadF@primaryAASeqS2), "WLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSF")
    expect_equal(as.character(trimmedsangerReadF@primaryAASeqS3), "GWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHL")
    expect_equal(trimmedsangerReadF@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF@QualityReport@TrimmingMethod, "M1")
    expect_equal(trimmedsangerReadF@QualityReport@M1TrimmingCutoff, 0.000001, tolerance=1e-10)
    expect_equal(trimmedsangerReadF@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(trimmedsangerReadF@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(trimmedsangerReadF@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF@QualityReport@trimmedSeqLength, 364)
    expect_equal(trimmedsangerReadF@QualityReport@trimmedStartPos, 34)
    expect_equal(trimmedsangerReadF@QualityReport@trimmedFinishPos, 398)
    expect_equal(trimmedsangerReadF@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF@QualityReport@trimmedMeanQualityScore, 58.99451, tolerance=1e-6)
    expect_equal(trimmedsangerReadF@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF@QualityReport@trimmedMinQualityScore, 15)
    expect_equal(trimmedsangerReadF@QualityReport@remainingRatio, 0.5185185, tolerance=1e-6)
})
test_that("sangerReadF update quality trimming parameters 2 (bigger M1)", {
    trimmedsangerReadF2 <- updateQualityParam(sangerReadF,
                                             TrimmingMethod       = "M1",
                                             M1TrimmingCutoff     = 0.1,
                                             M2CutoffQualityScore = NULL,
                                             M2SlidingWindowSize  = NULL)
    expect_type(trimmedsangerReadF2, "S4")
    expect_s4_class(trimmedsangerReadF2, "SangerRead")
    expect_equal(trimmedsangerReadF2@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF2@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF2@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF2@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF2@primaryAASeqS1), "YILFWASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEEDYVLNEFPCLSELC*LQLFFYFYLYQC*QVPLPYFLPTETSILHSLILPVVETPSSTSTYSDF*ITLMLSHTEF")
    expect_equal(as.character(trimmedsangerReadF2@primaryAASeqS2), "IFYSGRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKRITS*TNSPVCLSCANYSCSSTSIFTSASRCHYHTSYRPKPQYFIL*SCRWWRPHPLLALILIFRSP*C*VILNS")
    expect_equal(as.character(trimmedsangerReadF2@primaryAASeqS3), "YFILGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*RGLRLERIPLFV*AVLITVVLLLLSLPVLAGAITILLTDRNLNTSFFDPAGGGDPILY*HLF*FLDHPDVESY*I")
    expect_equal(trimmedsangerReadF2@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF2@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF2@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF2@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF2@QualityReport@TrimmingMethod, "M1")
    expect_equal(trimmedsangerReadF2@QualityReport@M1TrimmingCutoff, 0.1, tolerance=1e-10)
    expect_equal(trimmedsangerReadF2@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(trimmedsangerReadF2@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(trimmedsangerReadF2@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF2@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF2@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF2@QualityReport@trimmedSeqLength, 694)
    expect_equal(trimmedsangerReadF2@QualityReport@trimmedStartPos, 5)
    expect_equal(trimmedsangerReadF2@QualityReport@trimmedFinishPos, 699)
    expect_equal(trimmedsangerReadF2@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF2@QualityReport@trimmedMeanQualityScore, 53.38329, tolerance=1e-6)
    expect_equal(trimmedsangerReadF2@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF2@QualityReport@trimmedMinQualityScore, 5)
    expect_equal(trimmedsangerReadF2@QualityReport@remainingRatio, 0.988604, tolerance=1e-6)
})
test_that("sangerReadF update quality trimming parameters 3 (equal to 1). Only trim first base", {
    # First base will always be trimmed ~
    trimmedsangerReadF3 <- updateQualityParam(sangerReadF,
                                             TrimmingMethod       = "M1",
                                             M1TrimmingCutoff     = 1,
                                             M2CutoffQualityScore = NULL,
                                             M2SlidingWindowSize  = NULL)
    expect_type(trimmedsangerReadF3, "S4")
    expect_s4_class(trimmedsangerReadF3, "SangerRead")
    expect_equal(trimmedsangerReadF3@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF3@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF3@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF3@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF3@primaryAASeqS1), "TLYFILGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*RGLRLERIPLFV*AVLITVVLLLLSLPVLAGAITILLTDRNLNTSFFDPAGGGDPILY*HLF*FLDHPDVESY*IP")
    expect_equal(as.character(trimmedsangerReadF3@primaryAASeqS2), "LYILFWASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEEDYVLNEFPCLSELC*LQLFFYFYLYQC*QVPLPYFLPTETSILHSLILPVVETPSSTSTYSDF*ITLMLSHTEFL")
    expect_equal(as.character(trimmedsangerReadF3@primaryAASeqS3), "FIFYSGRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKRITS*TNSPVCLSCANYSCSSTSIFTSASRCHYHTSYRPKPQYFIL*SCRWWRPHPLLALILIFRSP*C*VILNS*")
    expect_equal(trimmedsangerReadF3@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF3@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF3@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF3@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF3@QualityReport@TrimmingMethod, "M1")
    expect_equal(trimmedsangerReadF3@QualityReport@M1TrimmingCutoff, 1, tolerance=1e-10)
    expect_equal(trimmedsangerReadF3@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(trimmedsangerReadF3@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(trimmedsangerReadF3@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF3@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF3@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF3@QualityReport@trimmedSeqLength, 701)
    expect_equal(trimmedsangerReadF3@QualityReport@trimmedStartPos, 1)
    expect_equal(trimmedsangerReadF3@QualityReport@trimmedFinishPos, 702)
    expect_equal(trimmedsangerReadF3@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF3@QualityReport@trimmedMeanQualityScore, 52.94864, tolerance=1e-6)
    expect_equal(trimmedsangerReadF3@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF3@QualityReport@trimmedMinQualityScore, 1)
    expect_equal(trimmedsangerReadF3@QualityReport@remainingRatio, 0.9985755, tolerance=1e-6)
})
test_that("sangerReadF update quality trimming parameters 4 (very small). All trimmed", {
    # First base will always be trimmed ~
    trimmedsangerReadF4 <- updateQualityParam(sangerReadF,
                                             TrimmingMethod       = "M1",
                                             M1TrimmingCutoff     =  0.000000001,
                                             M2CutoffQualityScore = NULL,
                                             M2SlidingWindowSize  = NULL)
    expect_type(trimmedsangerReadF4, "S4")
    expect_s4_class(trimmedsangerReadF4, "SangerRead")
    expect_equal(trimmedsangerReadF4@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF4@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF4@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF4@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF4@primaryAASeqS1), "")
    expect_equal(as.character(trimmedsangerReadF4@primaryAASeqS2), "")
    expect_equal(as.character(trimmedsangerReadF4@primaryAASeqS3), "")
    expect_equal(trimmedsangerReadF4@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF4@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF4@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF4@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF4@QualityReport@TrimmingMethod, "M1")
    expect_equal(trimmedsangerReadF4@QualityReport@M1TrimmingCutoff, 0.000000001, tolerance=1e-10)
    expect_equal(trimmedsangerReadF4@QualityReport@M2CutoffQualityScore, NULL)
    expect_equal(trimmedsangerReadF4@QualityReport@M2SlidingWindowSize, NULL)
    expect_equal(length(trimmedsangerReadF4@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF4@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF4@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF4@QualityReport@trimmedSeqLength, 1)
    expect_equal(trimmedsangerReadF4@QualityReport@trimmedStartPos, 1)
    expect_equal(trimmedsangerReadF4@QualityReport@trimmedFinishPos, 2)
    expect_equal(trimmedsangerReadF4@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF4@QualityReport@trimmedMeanQualityScore, 4, tolerance=1e-6)
    expect_equal(trimmedsangerReadF4@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF4@QualityReport@trimmedMinQualityScore, 4)
    expect_equal(trimmedsangerReadF4@QualityReport@remainingRatio, 0.001424501, tolerance=1e-6)
})
# test_that("sangerReadF update quality trimming parameters 5 (smaller than 0).", {
#     # First base will always be trimmed ~
#     expect_error(updateQualityParam(sangerReadF,
#                                     TrimmingMethod       = "M1",
#                                     M1TrimmingCutoff     =  -0.00001,
#                                     M2CutoffQualityScore = NULL,
#                                     M2SlidingWindowSize  = NULL),
#                  "Your input M1TrimmingCutoff is: '-1e-05' is invalid.'M1TrimmingCutoff' shouldbe between 0 and 1.", fixed = TRUE)
# })
# test_that("sangerReadF update quality trimming parameters 6 (bigger than 1).", {
#     # First base will always be trimmed ~
#     expect_error(updateQualityParam(sangerReadF,
#                                     TrimmingMethod       = "M1",
#                                     M1TrimmingCutoff     =  1.0001,
#                                     M2CutoffQualityScore = NULL,
#                                     M2SlidingWindowSize  = NULL),
#                  "Your input M1TrimmingCutoff is: '1.0001' is invalid.'M1TrimmingCutoff' shouldbe between 0 and 1.", fixed = TRUE)
# })
test_that("sangerReadF update quality trimming parameters 7 (Change from M1 to M2)", {
    trimmedsangerReadF7 <- updateQualityParam(sangerReadF,
                                             TrimmingMethod       = "M2",
                                             M1TrimmingCutoff     = NULL,
                                             M2CutoffQualityScore = 25,
                                             M2SlidingWindowSize  = 10)
    expect_type(trimmedsangerReadF7, "S4")
    expect_s4_class(trimmedsangerReadF7, "SangerRead")
    expect_equal(trimmedsangerReadF7@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF7@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF7@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF7@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF7@primaryAASeqS1), "YFILGV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*RGLRLERIPLFV*AVLITVVLLLLSLPVLAGAITILLTDRNLNTSFFDPAGGGDPILY*HLF*FLDHPDVES")
    expect_equal(as.character(trimmedsangerReadF7@primaryAASeqS2), "ILFWASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEEDYVLNEFPCLSELC*LQLFFYFYLYQC*QVPLPYFLPTETSILHSLILPVVETPSSTSTYSDF*ITLMLS")
    expect_equal(as.character(trimmedsangerReadF7@primaryAASeqS3), "FYSGRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKRITS*TNSPVCLSCANYSCSSTSIFTSASRCHYHTSYRPKPQYFIL*SCRWWRPHPLLALILIFRSP*C*V")
    expect_equal(trimmedsangerReadF7@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF7@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF7@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF7@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF7@QualityReport@TrimmingMethod, "M2")
    expect_equal(trimmedsangerReadF7@QualityReport@M1TrimmingCutoff, NULL, tolerance=1e-10)
    expect_equal(trimmedsangerReadF7@QualityReport@M2CutoffQualityScore, 25, tolerance=1e-10)
    expect_equal(trimmedsangerReadF7@QualityReport@M2SlidingWindowSize, 10, tolerance=1e-10)
    expect_equal(length(trimmedsangerReadF7@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF7@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF7@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF7@QualityReport@trimmedSeqLength, 681)
    expect_equal(trimmedsangerReadF7@QualityReport@trimmedStartPos, 7)
    expect_equal(trimmedsangerReadF7@QualityReport@trimmedFinishPos, 688)
    expect_equal(trimmedsangerReadF7@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF7@QualityReport@trimmedMeanQualityScore, 54.05286, tolerance=1e-6)
    expect_equal(trimmedsangerReadF7@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF7@QualityReport@trimmedMinQualityScore, 8)
    expect_equal(trimmedsangerReadF7@QualityReport@remainingRatio, 0.9700855, tolerance=1e-6)
})
test_that("sangerReadF update quality trimming parameters 8 (Update from M1 to M2)", {
    trimmedsangerReadF <- updateQualityParam(sangerReadF,
                                            TrimmingMethod       = "M1",
                                            M1TrimmingCutoff     = 0.000001,
                                            M2CutoffQualityScore = NULL,
                                            M2SlidingWindowSize  = NULL)
    trimmedsangerReadF8 <- updateQualityParam(trimmedsangerReadF,
                                             TrimmingMethod       = "M2",
                                             M1TrimmingCutoff     = NULL,
                                             M2CutoffQualityScore = 30,
                                             M2SlidingWindowSize  = 12)
    expect_type(trimmedsangerReadF8, "S4")
    expect_s4_class(trimmedsangerReadF8, "SangerRead")
    expect_equal(trimmedsangerReadF8@readFeature, "Forward Read")
    expect_equal(basename(trimmedsangerReadF8@readFileName), "Achl_ACHLO006-09_1_F.ab1")
    expect_equal(as.character(trimmedsangerReadF8@primarySeq), "CACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTAGATCACCCTGATGTTGAGTCATACTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF8@secondarySeq), "TACTTTATATTTTATTCTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCTGGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCATCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGATTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGTTTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTGCGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGGATTACGTCTTGAACGAATTCCCCTGTTTGTCTGAGCTGTGCTAATTACAGTTGTTCTTCTACTTCTATCTTTACCAGTGCTAGCAGGTGCCATTACCATACTTCTTACCGACCGAAACCTCAATACTTCATTCTTTGATCCTGCCGGTGGTGGAGACCCCATCCTCTACTAGCACTTATTCTGATTTTTCGATCACCCTGATGTTGAGTCATAGTGAATTCCTGA")
    expect_equal(as.character(trimmedsangerReadF8@primaryAASeqS1),  "GRLSRNGWSRYKTSNSNRAKTTRSVPGQRPTIQYYRYCTRICNNLLSSNACIHRGIRKLAFTFNTWSPRYSIPSTQQHEILTTSPITDPFSVLCGGRKRRWYGVNCLSASSKKSCPRRPVCRFSHLFPSFSGCVFYSRGY*FYHHSY*YALKRITS*TNSPVCLSCANYSCSSTSIFTSASRCHYHTSYRPKPQYFIL*SCRWWRPHPLLALILIFRSP*C*V")
    expect_equal(as.character(trimmedsangerReadF8@primaryAASeqS2), "GV*AGMVGAGIRLLIRIELRQPGAFLGRDQLYNTIVTAHAFVIIFFLVMPVFIGGFGNWLLPLILGAPDIAFPRLNNMRF*LLPPSLILLVSSAAVEKGAGTG*TVYPPLARNLAHAGPSVDLAIFSLHLAGASSILGAINFITTVINMR*RGLRLERIPLFV*AVLITVVLLLLSLPVLAGAITILLTDRNLNTSFFDPAGGGDPILY*HLF*FLDHPDVE")
    expect_equal(as.character(trimmedsangerReadF8@primaryAASeqS3), "ASEQEWLEPV*DF*FESS*DNQERSWAETNYTILSLLHTHL**SSF**CLYSSGDSETGFYL*YLEPPI*HSLDSTT*DSDYFPHH*SF*CPLRR*KKALVRGELFIRL*QEILPTQARL*I*PSFPFI*RVRLLF*GLLILSPQLLICVEEDYVLNEFPCLSELC*LQLFFYFYLYQC*QVPLPYFLPTETSILHSLILPVVETPSSTSTYSDF*ITLMLS")
    expect_equal(trimmedsangerReadF8@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF8@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF8@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF8@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF8@QualityReport@TrimmingMethod, "M2")
    expect_equal(trimmedsangerReadF8@QualityReport@M1TrimmingCutoff, NULL)
    expect_equal(trimmedsangerReadF8@QualityReport@M2CutoffQualityScore, 30, tolerance=1e-10)
    expect_equal(trimmedsangerReadF8@QualityReport@M2SlidingWindowSize, 12, tolerance=1e-10)
    expect_equal(length(trimmedsangerReadF8@QualityReport@qualityPhredScores), 702)
    expect_equal(length(trimmedsangerReadF8@QualityReport@qualityBaseScores), 702)

    expect_equal(trimmedsangerReadF8@QualityReport@rawSeqLength, 702)
    expect_equal(trimmedsangerReadF8@QualityReport@trimmedSeqLength, 669)
    expect_equal(trimmedsangerReadF8@QualityReport@trimmedStartPos, 18)
    expect_equal(trimmedsangerReadF8@QualityReport@trimmedFinishPos, 687)
    expect_equal(trimmedsangerReadF8@QualityReport@rawMeanQualityScore, 52.87607, tolerance=1e-6)
    expect_equal(trimmedsangerReadF8@QualityReport@trimmedMeanQualityScore, 54.58894, tolerance=1e-6)
    expect_equal(trimmedsangerReadF8@QualityReport@rawMinQualityScore, 1)
    expect_equal(trimmedsangerReadF8@QualityReport@trimmedMinQualityScore, 12)
    expect_equal(trimmedsangerReadF8@QualityReport@remainingRatio, 0.9529915, tolerance=1e-6)
})
# test_that("sangerReadF update quality trimming parameters 9 (M2CutoffQualityScore bigger than threashold)", {
#     expect_error(updateQualityParam(sangerReadF,
#                                     TrimmingMethod       = "M2",
#                                     M1TrimmingCutoff     = NULL,
#                                     M2CutoffQualityScore = 61,
#                                     M2SlidingWindowSize  = 41),
#                  "\nYour input M2CutoffQualityScore is: '61' is invalid.'M2CutoffQualityScore' shouldbe between 0 and 60.\n\nYour input M2SlidingWindowSize is: '41' is invalid.'M2SlidingWindowSize' shouldbe between 0 and 40.\n", fixed = TRUE)
# })
# test_that("sangerReadF update quality trimming parameters 10 (M2CutoffQualityScore smaller than threashold)", {
#     expect_error(updateQualityParam(sangerReadF,
#                                     TrimmingMethod       = "M2",
#                                     M1TrimmingCutoff     = NULL,
#                                     M2CutoffQualityScore = -1,
#                                     M2SlidingWindowSize  = -1),
#                  "\nYour input M2CutoffQualityScore is: '-1' is invalid.'M2CutoffQualityScore' shouldbe between 0 and 60.\n\nYour input M2SlidingWindowSize is: '-1' is invalid.'M2SlidingWindowSize' shouldbe between 0 and 40.\n", fixed = TRUE)
# })
test_that("sangerReadF update quality trimming parameters 11 (Update from M1 to M2) - reverse", {
    trimmedsangerReadR <- updateQualityParam(sangerReadR,
                                             TrimmingMethod       = "M1",
                                             M1TrimmingCutoff     = 0.000001,
                                             M2CutoffQualityScore = NULL,
                                             M2SlidingWindowSize  = NULL)
    trimmedsangerReadF11 <- updateQualityParam(trimmedsangerReadR,
                                               TrimmingMethod       = "M2",
                                               M1TrimmingCutoff     = NULL,
                                               M2CutoffQualityScore = 30,
                                               M2SlidingWindowSize  = 12)
    expect_type(trimmedsangerReadF11, "S4")
    expect_s4_class(trimmedsangerReadF11, "SangerRead")
    expect_equal(trimmedsangerReadF11@readFeature, "Reverse Read")
    expect_equal(basename(trimmedsangerReadF11@readFileName), "Achl_ACHLO006-09_2_R.ab1")
    expect_equal(as.character(trimmedsangerReadF11@primarySeq), "CCGTAGTAGGTGTTGGTAGAGGATGGGGTCTCCACCACCGGCAGGATCAAAGAATGAAGTATTGAGGTTTCGGTCGGTAAGAAGTATGGTAATGGCACCTGCTAGCACTGGTAAAGATAGAAGTAGAAGAACAACTGTAATTAGCACAGCTCAGACAAACAGGGGAATTCGTTCAAGACGTAATCCTCTTCAACGCATATTAATAACTGTGGTGATAAAATTAATAGCCCCTAGAATAGAAGACGCACCCGCTAAATGAAGGGAAAAGATGGCTAAATCTACAGACGGGCCTGCGTGGGCAAGATTTCTTGCTAGAGGCGGATAAACAGTTCACCCCGTACCAGCGCCTTTTTCTACCGCCGCAGAGGACACTAAAAGGATCAGTGATGGGGGAAGTAGTCAGAATCTCATGTTGTTGAGTCGAGGGAATGCTATATCGGGGGCTCCAAGTATTAAAGGTAAAAGCCAGTTTCCGAATCCCCCGATGAATACAGGCATTACTAGAAAGAAGATTATTACAAATGCGTGTGCAGTAACGATAGTATTGTATAGTTGGTCTCTGCCCAGGAACGCTCCTGGTTGTCTTAGCTCGATTCGAATTAGAAGTCTTATACCGGCTCCAACCATTCCTGCTCAGACGCCCAGAATAAAATATAAAGTTCCAATATCTTTATGATTTGTTGACCACTGGCCGTCGATATACAA")
    expect_equal(as.character(trimmedsangerReadF11@secondarySeq), "CAGCAGTAGGTGTTGGTAGAGGATGGGGTCTCCACCACCGGCAGGATCAAAGAATGAAGTATTGCGGTTTCGGTCGGTAAGAAGTATGGTAATGGCACCTGCTAGCACTGGTAAAGATAGAAGTAGAAGAACAACTGTAATTAGCACAGCTCAGACAAACAGGGGAATTCGTTCAAGACGTAATCCTCTTCAACGCATATTAATAACTGTGGTGATAAAATTAATAGCCCCTAGAATAGAAGACGCACCCGCTAAATGAAGGGAAAAGATGGCTAAATCTACAGACGGGCCTGCGTGGGCAAGATTTCTTGCTAGAGGCGGATAAACAGTTCACCCCGTACCAGCGCCTTTTTCTACCGCCGCAGAGGACACTAAAAGGATCAGTGATGGGGGAAGTAGTCAGAATCTCATGTTGTTGAGTCGAGGGAATGCTATATCGGGGGCTCCAAGTATTAAAGGTAAAAGCCAGTTTCCGAATCCCCCGATGAATACAGGCATTACTAGAAAGAAGATTATTACAAATGCGTGTGCAGTAACGATAGTATTGTATAGTTGGTCTCTGCCCAGGAACGCTCCTGGTTGTCTTAGCTCGATTCGAATTAGAAGTCTTATACCGGCTCCAACCATTCCTGCTCAGACGCCCAGAATAAAATATAAAGTTCCAATATCTTTATGATTTGTTGACCACTGGCCGTCGATGTACAA")
    expect_equal(as.character(trimmedsangerReadF11@primaryAASeqS1),  "GWGLHHRQDQRMKY*GFGR*EVW*WHLLALVKIEVEEQL*LAQLRQTGEFVQDVILFNAY**LW**N**PLE*KTHPLNEGKRWLNLQTGLRGQDFLLEADKQFTPYQRLFLPPQRTLKGSVMGEVVRISCC*VEGMLYRGLQVLKVKASFRIPR*IQALLERRLLQMRVQ*R*YCIVGLCPGTLLVVLARFELEVLYRLQPFLLRRPE*NIKFQYLYDLLTTGRRYT")
    expect_equal(as.character(trimmedsangerReadF11@primaryAASeqS2), "DGVSTTGRIKE*SIEVSVGKKYGNGTC*HW*R*K*KNNCN*HSSDKQGNSFKT*SSSTHINNCGDKINSP*NRRRTR*MKGKDG*IYRRACVGKISC*RRINSSPRTSAFFYRRRGH*KDQ*WGK*SESHVVESRECYIGGSKY*R*KPVSESPDEYRHY*KEDYYKCVCSNDSIV*LVSAQERSWLS*LDSN*KSYTGSNHSCSDAQNKI*SSNIFMIC*PLAVDI")
    expect_equal(as.character(trimmedsangerReadF11@primaryAASeqS3), "MGSPPPAGSKNEVLRFRSVRSMVMAPASTGKDRSRRTTVISTAQTNRGIRSRRNPLQRILITVVIKLIAPRIEDAPAK*REKMAKSTDGPAWARFLARGG*TVHPVPAPFSTAAEDTKRISDGGSSQNLMLLSRGNAISGAPSIKGKSQFPNPPMNTGITRKKIITNACAVTIVLYSWSLPRNAPGCLSSIRIRSLIPAPTIPAQTPRIKYKVPISL*FVDHWPSIY")
    expect_equal(trimmedsangerReadF11@ChromatogramParam@baseNumPerRow, 100)
    expect_equal(trimmedsangerReadF11@ChromatogramParam@heightPerRow, 200)
    expect_equal(trimmedsangerReadF11@ChromatogramParam@signalRatioCutoff, 0.33)
    expect_equal(trimmedsangerReadF11@ChromatogramParam@showTrimmed, TRUE)

    expect_equal(trimmedsangerReadF11@QualityReport@TrimmingMethod, "M2")
    expect_equal(trimmedsangerReadF11@QualityReport@M1TrimmingCutoff, NULL)
    expect_equal(trimmedsangerReadF11@QualityReport@M2CutoffQualityScore, 30, tolerance=1e-10)
    expect_equal(trimmedsangerReadF11@QualityReport@M2SlidingWindowSize, 12, tolerance=1e-10)
    expect_equal(length(trimmedsangerReadF11@QualityReport@qualityPhredScores), 705)
    expect_equal(length(trimmedsangerReadF11@QualityReport@qualityBaseScores), 705)

    expect_equal(trimmedsangerReadF11@QualityReport@rawSeqLength, 705)
    expect_equal(trimmedsangerReadF11@QualityReport@trimmedSeqLength, 684)
    expect_equal(trimmedsangerReadF11@QualityReport@trimmedStartPos, 20)
    expect_equal(trimmedsangerReadF11@QualityReport@trimmedFinishPos, 704)
    expect_equal(trimmedsangerReadF11@QualityReport@rawMeanQualityScore, 54.85106, tolerance=1e-6)
    expect_equal(trimmedsangerReadF11@QualityReport@trimmedMeanQualityScore, 56.14766, tolerance=1e-6)
    expect_equal(trimmedsangerReadF11@QualityReport@rawMinQualityScore, 3)
    expect_equal(trimmedsangerReadF11@QualityReport@trimmedMinQualityScore, 6)
    expect_equal(trimmedsangerReadF11@QualityReport@remainingRatio, 0.9702128 , tolerance=1e-6)
})

### ============================================================================
### sangerReadF ABIF Write FASTA
### ============================================================================
test_that("sangerReadF writeFastaSR", {
    outputFasta <- writeFastaSR(sangerReadF)
    expect_true(file.exists(outputFasta))
    con = file(outputFasta, "r")
    line = readLines(con, n = 1)
    expect_equal(line, ">Achl_ACHLO006-09_1_F.ab1")
    line = readLines(con, n = 1)
    expect_equal(line, "CTGGGCGTCTGAGCAGGAATGGTTGGAGCCGGTATAAGACTTCTAATTCGAATCGAGCTAAGACAACCAGGAGCGTTCCT")
    line = readLines(con, n = 1)
    expect_equal(line, "GGGCAGAGACCAACTATACAATACTATCGTTACTGCACACGCATTTGTAATAATCTTCTTTCTAGTAATGCCTGTATTCA")
    line = readLines(con, n = 1)
    expect_equal(line, "TCGGGGGATTCGGAAACTGGCTTTTACCTTTAATACTTGGAGCCCCCGATATAGCATTCCCTCGACTCAACAACATGAGA")
    line = readLines(con, n = 1)
    expect_equal(line, "TTCTGACTACTTCCCCCATCACTGATCCTTTTAGTGTCCTCTGCGGCGGTAGAAAAAGGCGCTGGTACGGGGTGAACTGT")
    line = readLines(con, n = 1)
    expect_equal(line, "TTATCCGCCTCTAGCAAGAAATCTTGCCCACGCAGGCCCGTCTGTAGATTTAGCCATCTTTTCCCTTCATTTAGCGGGTG")
    line = readLines(con, n = 1)
    expect_equal(line, "CGTCTTCTATTCTAGGGGCTATTAATTTTATCACCACAGTTATTAATATGCGTTGAAGAGG")
    close(con)
    unlink(outputFasta)
})

