% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simplifyTCGA.R
\name{simplifyTCGA}
\alias{simplifyTCGA}
\alias{symbolsToRanges}
\alias{CpGtoRanges}
\alias{qreduceTCGA}
\title{Functions to convert rows annotations to ranges and RaggedExperiment
to RangedSummarizedExperiment}
\usage{
simplifyTCGA(obj, keep.assay = FALSE, unmapped = TRUE)

symbolsToRanges(obj, keep.assay = FALSE, unmapped = TRUE)

CpGtoRanges(obj, keep.assay = FALSE, unmapped = TRUE)

qreduceTCGA(obj, keep.assay = FALSE, suffix = "_simplified")
}
\arguments{
\item{obj}{A \code{MultiAssayExperiment} object obtained from \code{curatedTCGAData}}

\item{keep.assay}{logical (default FALSE) Whether to keep the
\code{SummarizedExperiment} assays that have been converted to
\code{RangedSummarizedExperiment}}

\item{unmapped}{logical (default TRUE) Include an assay of data that was
not able to be mapped in reference database}

\item{suffix}{character (default "_simplified") A character string to append
to the newly modified assay for \code{qreduceTCGA}.}
}
\value{
A
\code{\link[MultiAssayExperiment:MultiAssayExperiment-class]{MultiAssayExperiment}}
with any gene expression, miRNA, copy number, and mutations converted to
\code{\link[SummarizedExperiment:RangedSummarizedExperiment-class]{RangedSummarizedExperiment}}
objects
}
\description{
This group of functions will convert row annotations as
either gene symbols or miRNA symbols to row ranges based on database
resources 'TxDB' and 'org.Hs' packages. It will also simplify the
representation of
\link[RaggedExperiment:RaggedExperiment-class]{RaggedExperiment} objects to
\link[SummarizedExperiment:RangedSummarizedExperiment-class]{RangedSummarizedExperiment}.
}
\details{
The original \code{SummarizedExperiment} containing either gene symbol
or miR annotations is replaced or supplemented by a
\link[SummarizedExperiment:RangedSummarizedExperiment-class]{RangedSummarizedExperiment}
for those that could be mapped to
\link[GenomicRanges:GRanges-class]{GRanges}, and optionally another
\link[SummarizedExperiment:SummarizedExperiment-class]{SummarizedExperiment}
for annotations that could not be mapped to
\link[GenomicRanges:GRanges-class]{GRanges}.
}
\section{qreduceTCGA}{


Using \code{TxDb.Hsapiens.UCSC.hg19.knownGene} as the reference, \code{qreduceTCGA}
reduces the data by applying either the \code{weightedmean} or \code{nonsilent}
function (see below) to non-mutation or mutation data, respectively.
Internally, it uses \code{\link[RaggedExperiment:assay-functions]{RaggedExperiment::qreduceAssay()}} to reduce the ranges
to the gene-level.

\code{qreduceTCGA} will update \code{genome(x)} based on the NCBI reference annotation
which includes the patch number, e.g., GRCh37.p14, as provided by the
\code{seqlevelsStyle} setter, \code{seqlevelsStyle(gn) <- "NCBI"}. \code{qreduceTCGA}
uses the NCBI genome annotation as the default reference.

\if{html}{\out{<div class="sourceCode">}}\preformatted{nonsilent <- function(scores, ranges, qranges)
    any(scores != "Silent")
}\if{html}{\out{</div>}}

\code{RaggedExperiment} mutation objects become a genes by patients
\code{RangedSummarizedExperiment} object containing '1' if there is a non-silent
mutation somewhere in the gene, and '0' otherwise as obtained from the
\code{Variant_Classification} column in the data.

\if{html}{\out{<div class="sourceCode">}}\preformatted{weightedmean <- function(scores, ranges, qranges) \{
    isects <- GenomicRanges::pintersect(ranges, qranges)
    sum(scores * BiocGenerics::width(isects)) /
        sum(BiocGenerics::width(isects))
\}
}\if{html}{\out{</div>}}

"CNA" and "CNV" segmented copy number are reduced using a weighted mean in
the rare cases of overlapping (non-disjoint) copy number regions.

These functions rely on \code{TxDb.Hsapiens.UCSC.hg19.knownGene} and
\code{org.Hs.eg.db} to map to the 'hg19' NCBI build. Use the \code{liftOver} procedure
for datasets that are provided against a different reference genome (usually
'hg18'). See an example in the vignette.
}

\examples{

library(curatedTCGAData)
library(GenomeInfoDb)

accmae <-
    curatedTCGAData(diseaseCode = "ACC",
    assays = c("CNASNP", "Mutation", "miRNASeqGene", "GISTICT"),
    version = "1.1.38",
    dry.run = FALSE)

## update genome annotation
rex <- accmae[["ACC_Mutation-20160128"]]

## Translate build to "hg19"
tgenome <- vapply(genome(rex), translateBuild, character(1L))
genome(rex) <- tgenome

accmae[["ACC_Mutation-20160128"]] <- rex

simplifyTCGA(accmae)

}
\author{
L. Waldron
}
