% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotCoverage.R
\name{plotCoverage}
\alias{plotCoverage}
\alias{plotGenome}
\title{RNA-seq Coverage Plots and Genome Tracks}
\usage{
plotCoverage(
  se,
  Event,
  Gene,
  seqname,
  start,
  end,
  coordinates,
  strand = c("*", "+", "-"),
  zoom_factor = 0.2,
  bases_flanking = 100,
  tracks,
  track_names = tracks,
  condition,
  ribbon_mode = c("sd", "ci", "sem", "none"),
  selected_transcripts = "",
  reverseGenomeCoords = FALSE,
  plotJunctions = FALSE,
  junctionThreshold = 0.01,
  plot_key_isoforms = FALSE,
  condense_tracks = FALSE,
  stack_tracks = FALSE,
  t_test = FALSE,
  norm_event,
  usePlotly = FALSE
)

plotGenome(
  se,
  reference_path,
  Event,
  Gene,
  seqname,
  start,
  end,
  coordinates,
  zoom_factor = 0.2,
  bases_flanking = 100,
  reverseGenomeCoords = FALSE,
  condense_tracks = FALSE,
  selected_transcripts = "",
  plot_key_isoforms = FALSE,
  usePlotly = FALSE
)
}
\arguments{
\item{se}{A \linkS4class{NxtSE} object, created by \link{makeSE}.
COV files must be linked to the NxtSE object. To do this, see the example
in \link{makeSE}. Required by \code{plotCoverage}. Not required by \code{plotGenome} if
\code{reference_path} is supplied.}

\item{Event}{The \code{EventName} of the IR / alternative splicing event to be
displayed. Use \code{rownames(se)} to display a list of valid events.}

\item{Gene}{Whether to use the range for the given \code{Gene}. If given,
overrides \code{Event} (but \code{Event} or \code{norm_event} will be used to normalise by
condition). Valid \code{Gene} entries include gene_id (Ensembl ID) or gene_name
(Gene Symbol).}

\item{seqname, start, end}{The chromosome (string) and genomic \code{start/end}
coordinates (numeric) of the region to display. If present, overrides both
\code{Event} and \code{Gene}. E.g. for a given region of chr1:10000-11000,
use the parameters: \verb{seqname = "chr1", start = 10000, end = 11000}}

\item{coordinates}{A string specifying genomic coordinates can be given
instead of \verb{seqname,start,end}. Must be of the format "chr:start-end", e.g.
"chr1:10000-11000"}

\item{strand}{Whether to show coverage of both strands "*" (default), or
from the "+" or "-" strand only.}

\item{zoom_factor}{Zoom out from event. Each level of zoom zooms out by a
factor of 3. E.g. for a query region of chr1:10000-11000, if a
\code{zoom_factor} of 1.0 is given, chr1:99000-12000 will be displayed.}

\item{bases_flanking}{(Default = \code{100}) How many bases flanking the zoomed
window. Useful when
used in conjunction with zoom_factor == 0. E.g. for a given region of
chr1:10000-11000, if \code{zoom_factor = 0} and \code{bases_flanking = 100}, the
region chr1:9900-11100 will be displayed.}

\item{tracks}{The names of individual samples,
or the names of the different conditions to be plotted. For the latter, set
\code{condition} to the specified condition category.}

\item{track_names}{The names of the tracks to be displayed. If omitted, the
track_names will default to the input in \code{tracks}}

\item{condition}{To display normalised coverage per condition, set this to
the condition category. If omitted, \code{tracks} are assumed to refer to the
names of individual samples.}

\item{ribbon_mode}{(default \code{"sd"}) Whether coverage ribbons signify
standard deviation \code{"sd"}, 95\% confidence interval \code{"ci"},
standard error of the mean \code{"sem"},
or none \code{"none"}. Only applicable when \code{condition} is set.}

\item{selected_transcripts}{(Optional) A vector containing transcript ID
or transcript names of transcripts
to be displayed on the gene annotation track. Useful to remove minor
isoforms that are not relevant to the samples being displayed.}

\item{reverseGenomeCoords}{(default \code{FALSE}) Whether to reverse the genomic
coordinates - helpful for intuitive plotting of negative-strand genes}

\item{plotJunctions}{(default \code{FALSE}) If \code{TRUE}, sashimi plot junction arcs
are plotted. Currently only implemented for plots of individual samples.}

\item{junctionThreshold}{(default \code{0.01}) The threshold expression of
junction reads below which junction arcs will be omitted. This removes
cluttering of junction arcs from lowly-expressed (rare) junctions.
For individual
tracks, this is the fraction of coverage height. For by-condition
tracks, this is a PSI threshold.}

\item{plot_key_isoforms}{(default \code{FALSE}) If \code{TRUE}, only
transcripts involved in the selected \code{Event} or pair of \code{Event}s will be
displayed.}

\item{condense_tracks}{(default \code{FALSE}) Whether to collapse the
transcript track annotations by gene.}

\item{stack_tracks}{(default \code{FALSE}) Whether to graph all the conditions on
a single coverage track. If set to \code{TRUE}, each condition will be displayed
in a different colour on the same track. Ignored if \code{condition} is not set.}

\item{t_test}{(default \code{FALSE}) Whether to perform a pair-wise T-test.
Only used if there are TWO condition tracks.}

\item{norm_event}{Whether to normalise by an event different to that given
in "Event". The difference between this and Event is that the genomic
coordinates can be centered around a different \code{Event}, \code{Gene} or region
as given in \code{seqname/start/end}. If \code{norm_event} is different to
\code{Event}, \code{norm_event} will be used for normalisation and \code{Event} will be
used to define the genomic coordinates of the viewing window. \code{norm_event}
is required if \code{Event} is not set and \code{condition} is set.}

\item{usePlotly}{If \code{TRUE}, returns a \code{covPlotly} object containing the
plotly-based interactive plot. If \code{FALSE}, returns a ggplot object.}

\item{reference_path}{The path of the reference generated by
\link{Build-Reference-methods}. Required by \code{plotGenome} if a
\linkS4class{NxtSE} object is not specified.}
}
\value{
For \code{plotCoverage} and \code{plotGenome}:
\itemize{
\item If \code{usePlotly = FALSE} returns a patchwork-assembled static plot
\item If \code{usePlotly = TRUE} returns a \linkS4class{covPlotly} object,
which generates a plotly interactive plot when shown using \code{show()}
}
}
\description{
Generate plotly / ggplot RNA-seq genome and coverage plots from command line.
Note that these are legacy functions. More expansive functionality is
available using \link{getCoverageData} / \link{getPlotObject} / \link{plotView} functions.
}
\details{
In RNA sequencing, alignments to spliced transcripts will "skip" over genomic
regions of introns. This can be illustrated in a plot using a horizontal
genomic axis, with the vertical axis representing the number of alignments
covering each nucleotide. As a result, the coverage "hills" represent the
expression of exons, and "valleys" to introns.

Different alternatively-spliced isoforms thus produce different coverage
patterns. The change in the coverage across an alternate exon relative to its
constitutively-included flanking exons, for example, represents its
alternative inclusion or skipping.  Similarly, elevation of intron
valleys represent increased intron retention.

With multiple replicates per sample, coverage is dependent on
library size and gene expression. To compare
alternative splicing ratios, normalisation of the coverage of the alternate
exon (or alternatively retained intron) relative to their constitutive
flanking exons, is required. There is no established method for this
normalisation, and can be confounded in situations where flanking elements
are themselves alternatively spliced.

SpliceWiz performs this coverage normalisation using the same method as its
estimate of spliced / intronic transcript abundance using the \code{SpliceOver}
method (see details section in \link{collateData}). This normalisation can be
applied to correct for library size and gene expression differences between
samples of the same experimental condition. After normalisation, mean and
variance of coverage can be computed as ratios relative to total transcript
abundance. This method can visualise alternatively included genomic regions
including casette exons, alternate splice site usage, and intron retention.

\code{plotCoverage} generates plots showing depth of alignments to
the genomic axis. Plots can be generated for individual samples or samples
grouped by experimental conditions. In the latter, mean and 95\% confidence
intervals are shown.

\code{plotGenome} generates genome transcript tracks only. Protein-coding
regions are denoted by thick rectangles, whereas non-protein coding
transcripts or untranslated regions are denoted with thin rectangles.
Introns are denoted as lines.
}
\section{Functions}{
\itemize{
\item \code{plotCoverage()}: Legacy function - works by internally calling
getCoverageData(), getPlotObject(), then plotView()

\item \code{plotGenome()}: Legacy function - works by internally calling
getGenomeData(), followed by plotAnnoTrack()

}}
\examples{
se <- SpliceWiz_example_NxtSE(novelSplicing = TRUE)

# Assign annotation of the experimental conditions
colData(se)$treatment <- rep(c("A", "B"), each = 3)

# Verify that the COV files are linked to the NxtSE object:
covfile(se)

# Plot the genome track only, with specified gene:
plotGenome(se, Gene = "SRSF3")

# View the genome track, specifying a genomic region via coordinates:
plotGenome(se, coordinates = "chrZ:10000-20000")

# Return a list of ggplot and plotly objects, also plotting junction counts
plotCoverage(
    se = se,
    Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
    tracks = colnames(se)[1:4], plotJunctions = TRUE
)

# Plot the same, but as a plotly interactive plot

if(interactive()) {
    p <- plotCoverage(
        se = se,
        Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
        tracks = colnames(se)[1:4], plotJunctions = TRUE,
        usePlotly = TRUE
    )
    show(p)
}

# Plot by condition "treatment", including provisional PSIs
plotCoverage(
    se = se,
    Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
    tracks = c("A", "B"), condition = "treatment", plotJunctions = TRUE
)

# As above, but stack all traces into the same track
# - NB: plotJunctions is disabled when `stack_tracks = TRUE`
plotCoverage(
    se = se,
    Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
    tracks = c("A", "B"), condition = "treatment", stack_tracks = TRUE
)

# Plot the above, but unstancked, and with t-test track
# - NB: plotJunctions is disabled when `stack_tracks = TRUE`
plotCoverage(
    se = se,
    Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
    tracks = c("A", "B"), condition = "treatment", t_test = TRUE
)

# Select only transcripts involved in the selected alternative splicing event
plotCoverage(
    se = se,
    Event = "SE:SRSF3-203-exon4;SRSF3-202-int3",
    tracks = colnames(se)[1:4],
    plot_key_isoforms = TRUE
)
}
