% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuildRef_GO.R
\name{Gene-ontology-methods}
\alias{Gene-ontology-methods}
\alias{goASE}
\alias{goGenes}
\alias{extract_gene_ids_for_GO}
\alias{subset_EventNames_by_GO}
\alias{plotGO}
\title{Gene ontology (over-representation) analysis using enriched genes of
top alternative splicing events}
\usage{
goASE(
  enrichedEventNames,
  universeEventNames = NULL,
  se,
  ontologyType = c("BP", "MF", "CC"),
  pAdjustMethod = c("BH", "holm", "hochberg", "hommel", "bonferroni", "BY", "fdr",
    "none"),
  ontologyRef = NULL,
  ...
)

goGenes(
  enrichedGenes,
  universeGenes = NULL,
  ontologyRef,
  ontologyType = c("BP", "MF", "CC"),
  pAdjustMethod = c("BH", "holm", "hochberg", "hommel", "bonferroni", "BY", "fdr",
    "none"),
  ...
)

extract_gene_ids_for_GO(enrichedEventNames, universeEventNames = NULL, se)

subset_EventNames_by_GO(EventNames, go_id, se)

plotGO(
  res_go = NULL,
  plot_x = c("log10FDR", "foldEnrichment", "nGenes"),
  plot_size = c("nGenes", "foldEnrichment", "log10FDR"),
  plot_color = c("foldEnrichment", "nGenes", "log10FDR"),
  filter_n_terms = 20,
  filter_padj = 1,
  filter_pvalue = 1,
  trim_go_term = 50
)
}
\arguments{
\item{enrichedEventNames}{A vector of \code{EventName}s. This is typically one
or more \code{EventName}s of differential ASEs}

\item{universeEventNames}{A vector of \code{EventName}s, typically the
\code{EventName}s of all ASEs that were tested. If left as \code{NULL}, all genes
are considered background genes.}

\item{se}{The NxtSE object containing the GO reference and the \code{EventName}s}

\item{ontologyType}{One of either \code{"BP"} - biological pathways, \code{"MF"} -
molecular function, or \code{"CC"} - cellular component.}

\item{pAdjustMethod}{The method for p-value adjustment for FDR.
See \code{?p.adjust}}

\item{ontologyRef}{A valid gene ontology reference. This can be generated
either using \code{viewGO(reference_path)} or \code{ref(se)$ontology}. This field
is required for \code{goGenes()} and optional for \code{goASE()}. See details.}

\item{...}{Additional arguments to be passed to \code{fgsea::fora()}}

\item{enrichedGenes}{A vector of \code{gene_id} representing the list of enriched
genes. To generate a list of valid \code{gene_id}, see \link{viewGenes}}

\item{universeGenes}{(default \code{NULL}) A vector of \code{gene_id} representing the
list of background genes.}

\item{EventNames, go_id}{In \code{subset_EventNames_by_GO()}, a vector of ASE
\code{EventName}s to subset against the given \code{go_id}.}

\item{res_go}{For \code{plotGO}, the gene ontology results data object returned
by the \code{goASE()} function.}

\item{plot_x, plot_size, plot_color}{What parameters should be plotted on
the x-axis, bubble-size, or bubble-color? Valid options are
\verb{c("log10FDR", "foldEnrichment", "nGenes"). Defaults are }"log10FDR", "nGenes", "foldEnrichment"` for x-axis, bubble size/color,
respectively}

\item{filter_n_terms}{(default \code{20}) How many top terms to plot.}

\item{filter_padj, filter_pvalue}{Whether given GO results should be filtered
by adjusted p value (FDR) or nominal p value, respectively, prior to plot}

\item{trim_go_term}{(default \code{50}) For long GO terms, description will be
trimmed by first n characters, where \code{trim_go_term = n}}
}
\value{
For \code{goASE()} and \code{goGenes()}, a data table containing the following:
\itemize{
\item go_id: Gene ontology ID
\item go_term: Gene ontology term
\item pval: Raw p values
\item padj: Adjusted p values
\item overlap: Number of enriched genes (of enriched ASEs)
\item size: Number of background genes (of background ASEs)
\item overlapGenes: A list of \code{gene_id}'s from genes of enriched ASEs
\item expected: The number of overlap genes expected by random
}

For \code{extract_gene_ids_for_GO()}, a list containing the following:
\itemize{
\item genes: A vector of enriched \code{gene_id}s
\item universe: A vector of background \code{gene_id}s
}

For \code{subset_EventNames_by_GO()}, a vector of all ASE \code{EventName}s belonging
to the given gene ontology \code{go_id}
}
\description{
Genes containing differential alternative splicing events (ASEs) may be
enriched in key functional pathways. This can be identified using a simple
over-representation analysis. Biologists can identify key pathways of
interest in order to focus on studying ASEs belonging to genes of functional
interest.
}
\details{
Users can perform GO analysis using either the GO annotation compiled via
building the SpliceWiz reference using \code{buildRef()} , or via a
custom-supplied gene ontology annotation. This is done by
supplying their own GO annotations as an argument to \code{ontologyRef}. This
should be coerceable to a \code{data.frame} containing the following columns:
\itemize{
\item \code{gene_id} Gene ID's matching that used by the SpliceWiz reference
\item \code{go_id} Gene ontology ID terms, of the form \code{GO:XXXXXXX}
}
}
\section{Functions}{
\itemize{
\item \code{goASE()}: Performs over-representation gene ontology
analysis using a given list of enriched / background ASEs

\item \code{goGenes()}: Performs GO analysis given the set of
enriched and (optionally) the background (universe) genes.

\item \code{extract_gene_ids_for_GO()}: Produces a list containing enriched and
universe \code{gene_id}s of given enriched and background ASE \code{EventName}s

\item \code{subset_EventNames_by_GO()}: Returns a list of ASEs enriched in a given
gene ontology category

\item \code{plotGO()}: Produces a lollipop plot based on the
given gene ontology results object

}}
\examples{
# Generate example reference with `Homo sapiens` gene ontology

ref_path <- file.path(tempdir(), "Reference_withGO")
buildRef(
    reference_path = ref_path,
    fasta = chrZ_genome(),
    gtf = chrZ_gtf(),
    ontologySpecies = "Homo sapiens"
)

# Perform GO analysis using first 1000 genes
ontology <- viewGO(ref_path)
allGenes <- sort(unique(ontology$gene_id))

exampleGeneID <- allGenes[1:1000]
exampleBkgdID <- allGenes

go_df <- goGenes(
    enrichedGenes = exampleGeneID, 
    universeGenes = exampleBkgdID, 
    ontologyRef = ontology
)

# Plots the top 12 GO terms

plotGO(go_df, filter_n_terms = 12)

# Below example code of how to use output of differential ASEs for GO analysis
# It will not work with the example dataset because the reference must be 
# either human / mouse, or a  valid `ontologySpecies` given to buildRef()
# We hope the example code is simple enough to understand for users to adapt
# to their own workflows.

\dontrun{

se <- SpliceWiz_example_NxtSE(novelSplicing = TRUE)

colData(se)$treatment <- rep(c("A", "B"), each = 3)

require("limma")
res_limma <- ASE_limma(se, "treatment", "A", "B")

# Perform gene ontology analysis of the first 10 differential ASEs

go_df <- goASE(
  enrichedEventNames = res_limma$EventName[1:10],
  universeEventNames = res_limma$EventName,
  se = se
)

# Return a list of all ASEs belonging to the top enriched category

GOsubset_EventName <- subset_EventNames_by_GO(
  EventNames = res_limma$EventName,
  go_id = go_df$go_id[1],
  se = se
)

# Return a list of all ASEs belonging to the top enriched category.
# - typically used if one wishes to export `gene_id` for use in other gene
#   ontology tools

gene_id_list <- extract_gene_ids_for_GO(
  enrichedEventNames = res_limma$EventName[1:10],
  universeEventNames = res_limma$EventName,
  se = se
)

}
}
\seealso{
\link{Build-Reference-methods} on how to generate gene ontology annotations\cr\cr
}
