% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smooth.R
\name{smooth}
\alias{smooth}
\alias{coefMA}
\alias{coefWMA}
\alias{coefSG}
\title{Smoothing}
\usage{
smooth(x, cf)

coefMA(hws)

coefWMA(hws)

coefSG(hws, k = 3L)
}
\arguments{
\item{x}{\code{numeric}, i.e. m/z values.}

\item{cf}{\code{matrix}, a coefficient matrix generated by \code{coefMA}, \code{coefWMA} or
\code{coefSG}.}

\item{hws}{\code{integer(1)}, half window size, the resulting window reaches from
\code{(i - hws):(i + hws)}.}

\item{k}{\code{integer(1)}, set the order of the polynomial used to calculate the
coefficients.}
}
\value{
\code{smooth}: A \code{numeric} of the same length as \code{x}.

\code{coefMA}: A \code{matrix} with coefficients for a simple moving average.

\code{coefWMA}: A \code{matrix} with coefficients for a weighted moving average.

\code{coefSG}: A \code{matrix} with \emph{Savitzky-Golay-Filter} coefficients.
}
\description{
This function smoothes a numeric vector.
}
\details{
For the Savitzky-Golay-Filter the \code{hws} should be smaller than
\emph{FWHM} of the peaks (full width at half maximum; please find details in
Bromba and Ziegler 1981).

In general the \code{hws} for the (weighted) moving average (\code{coefMA}/\code{coefWMA})
has to bemuch smaller than for the Savitzky-Golay-Filter to conserve the
peak shape.
}
\section{Functions}{
\itemize{
\item \code{coefMA()}: Simple Moving Average

This function calculates the coefficients for a simple moving average.

\item \code{coefWMA()}: Weighted Moving Average

This function calculates the coefficients for a weighted moving average with
weights depending on the distance from the center calculated as
\code{1/2^abs(-hws:hws)} with the sum of all weigths normalized to 1.

\item \code{coefSG()}: Savitzky-Golay-Filter

This function calculates the Savitzky-Golay-Coefficients. The additional
argument \code{k} controls the order of the used polynomial. If \code{k} is set to zero
it yield a simple moving average.

}}
\note{
The \code{hws} depends on the used method ((weighted) moving
average/Savitzky-Golay).
}
\examples{
x <- c(1:10, 9:1)
plot(x, type = "b", pch = 20)
cf <- list(MovingAverage = coefMA(2),
           WeightedMovingAverage = coefWMA(2),
           SavitzkyGolay = coefSG(2))
for (i in seq_along(cf)) {
    lines(smooth(x, cf[[i]]), col = i + 1, pch = 20, type = "b")
}
legend("bottom", legend = c("x", names(cf)), pch = 20,
       col = seq_len(length(cf) + 1))
}
\references{
A. Savitzky and M. J. Golay. 1964.
Smoothing and differentiation of data by simplified least squares procedures.
Analytical chemistry, 36(8), 1627-1639.

M. U. Bromba and H. Ziegler. 1981.
Application hints for Savitzky-Golay digital smoothing filters.
Analytical Chemistry, 53(11), 1583-1586.

Implementation based on:
Steinier, J., Termonia, Y., & Deltour, J. (1972). Comments on Smoothing and
differentiation of data by simplified least square procedure.
Analytical Chemistry, 44(11), 1906-1909.
}
\seealso{
Other noise estimation and smoothing functions: 
\code{\link{noise}()}
}
\author{
Sebastian Gibb, Sigurdur Smarason (weighted moving average)
}
\concept{noise estimation and smoothing functions}
