% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xCell2GetLineage.R
\name{xCell2GetLineage}
\alias{xCell2GetLineage}
\title{Identify Cell Type Lineage Dependencies}
\usage{
xCell2GetLineage(labels, outFile = NULL)
}
\arguments{
\item{labels}{A data frame with the following required columns:
\itemize{
  \item \code{"ont"}: Cell type ontology ID (e.g., \code{"CL:0000545"}). Use \code{NA} if unavailable.  
    Ontologies can be accessed via \href{https://www.ebi.ac.uk/ols4/ontologies/cl}{EBI Ontology Lookup Service (OLS)}  
    or the \link[ontologyIndex]{ontologyIndex} package.
  \item \code{"label"}: Cell type name (e.g., \code{"T-helper 1 cell"}).
  \item \code{"sample"}: Sample or cell identifier matching column names in the gene expression matrix.
  \item \code{"dataset"}: Dataset or subject source. Use a constant value if not applicable.
}}

\item{outFile}{Optional. Output file name for saving dependencies as a TSV file.
The file includes columns for \code{"ont"}, \code{"label"}, \code{"descendants"}, and \code{"ancestors"}.  
Suitable for manual inspection and refinement before use in downstream analyses.}
}
\value{
If \code{outFile} is:
  \itemize{
    \item \code{NULL}: Returns a list of dependencies for each cell type, with descendants and ancestors as components.
    \item Specified: Writes a TSV file and warns the user to inspect and validate results manually.
  }
}
\description{
Identifies cell type dependencies based on the Cell Ontology, including both descendants and ancestors for each cell type.  
Enables manual inspection and refinement of lineage relationships to improve biological accuracy in \code{xCell2} analyses.
}
\details{
The \code{xCell2GetLineage} function generates lineage relationships for cell types based on the Cell Ontology.  
These relationships refine lineage-based dependencies, improving the biological relevance of gene signatures.  
Users can:
\itemize{
  \item Use the generated TSV file for manual adjustments before training custom references via \code{\link{xCell2Train}}.
  \item Skip this step entirely, allowing \code{xCell2Train} to infer dependencies automatically.
}

If no ontology IDs (\code{"ont"}) are provided, the function outputs empty dependencies with a message for user guidance.

\strong{Relationship with Other Functions:}
\itemize{
  \item \code{\link{xCell2Train}}: Incorporates lineage relationships during reference training.
  \item \code{\link{xCell2Analysis}}: Uses trained references for enrichment analysis.
}
}
\examples{
# For detailed examples, see the xCell2 vignette.

library(xCell2)

# Load demo reference object
data(dice_demo_ref, package = "xCell2")

# Prepare labels data frame
dice_labels <- SummarizedExperiment::colData(dice_demo_ref)
dice_labels <- as.data.frame(dice_labels)
dice_labels$ont <- NA
dice_labels$sample <- colnames(dice_demo_ref)
dice_labels$dataset <- "DICE"

# Assign ontology IDs
dice_labels[dice_labels$label == "B cells", ]$ont <- "CL:0000236"
dice_labels[dice_labels$label == "Monocytes", ]$ont <- "CL:0000576"
dice_labels[dice_labels$label == "NK cells", ]$ont <- "CL:0000623"
dice_labels[dice_labels$label == "T cells, CD8+", ]$ont <- "CL:0000625"
dice_labels[dice_labels$label == "T cells, CD4+", ]$ont <- "CL:0000624"
dice_labels[dice_labels$label == "T cells, CD4+, memory", ]$ont <- "CL:0000897"

# Generate cell type lineage dependencies
xCell2::xCell2GetLineage(labels = dice_labels)

# Manually inspect and adjust saved dependencies for refined lineage relationships
# Use the adjusted file as input to xCell2Train via the `lineageFile` parameter.

}
\seealso{
\code{\link{xCell2Train}} for training custom references with lineage data.  
\code{\link{xCell2Analysis}} for enrichment analysis using trained references.  
\code{\link{AnnotationHub}} to access ontology data.  
\code{\link{ontologyIndex}} to programmatically explore ontologies.
}
\author{
Almog Angel and Dvir Aran
}
