test_that(
    "add_array_coords",
    {
        if (!exists("spe")) {
            spe <- spatialLIBD::fetch_data(type = "visiumStitched_brain_spe")
        }

        ########################################################################
        #   Prepare sample_info
        ########################################################################

        sample_info <- dplyr::tibble(
            group = "Br2719",
            capture_area = c("V13B23-283_A1", "V13B23-283_C1", "V13B23-283_D1")
        )
        #   Add 'spaceranger_dir' column
        sr_dir <- tempdir()
        temp <- unzip(
            spatialLIBD::fetch_data("visiumStitched_brain_spaceranger"),
            exdir = sr_dir
        )
        sample_info$spaceranger_dir <- file.path(
            sr_dir, sample_info$capture_area, "outs", "spatial"
        )

        #   Add Fiji-output-related columns
        fiji_dir <- tempdir()
        temp <- unzip(
            spatialLIBD::fetch_data("visiumStitched_brain_Fiji_out"),
            exdir = fiji_dir
        )
        sample_info$fiji_xml_path <- temp[grep("xml$", temp)]
        sample_info$fiji_image_path <- temp[grep("png$", temp)]

        ## Re-size images and add more information to the sample_info
        sample_info <- rescale_fiji_inputs(sample_info, out_dir = tempdir())

        spe_input_dir <- tempdir()
        prep_fiji_coords(sample_info, out_dir = spe_input_dir)
        prep_fiji_image(sample_info, out_dir = spe_input_dir)

        ########################################################################
        #   Tests
        ########################################################################

        #   Remove any colData columns that should be added by add_array_coords()
        added_cols_regex <- "^(array|pxl)_(row|col)(_in_fullres)?_(original|rounded)$|^euclidean_error$|^shared_neighbors$"
        spe_copy <- spe
        temp <- colnames(spe_copy)
        colData(spe_copy) <- colData(spe_copy) |>
            as_tibble() |>
            mutate(across(matches(added_cols_regex), ~NULL)) |>
            DataFrame()
        colnames(spe_copy) <- temp

        #   For the "LSAP" algorithm in particular, we need a small
        #   SpatialExperiment to get reasonable runtimes. Take just 20 spots
        #   from each capture_area
        small_keys <- colData(spe_copy) |>
            as_tibble() |>
            group_by(capture_area) |>
            mutate(a = array_row + array_col) |>
            arrange(a) |>
            slice_head(n = 20) |>
            pull(key)
        spe_small <- spe_copy[, spe_copy$key %in% small_keys]

        #   Similarly, this hack subsets the spots in the tissue_positions.csv
        #   file
        tissue_path <- file.path(spe_input_dir, unique(sample_info$group), "tissue_positions.csv")
        readr::read_csv(tissue_path, show_col_types = FALSE) |>
            filter(key %in% small_keys) |>
            readr::write_csv(tissue_path)

        for (algorithm in c("LSAP", "Euclidean")) {
            spe_new <- add_array_coords(
                spe_small, sample_info, spe_input_dir,
                calc_error_metrics = TRUE,
                algorithm = algorithm
            )

            #   8 columns should've been added, matching the specific naming
            #   pattern
            expect_equal(
                length(grep(added_cols_regex, colnames(colData(spe_new)))), 8
            )

            #   "Original" columns should actually have their original values
            expect_identical(spe_small$array_row, spe_new$array_row_original)
            expect_identical(spe_small$array_col, spe_new$array_col_original)
            expect_identical(
                spatialCoords(spe_small)[, "pxl_row_in_fullres"],
                spe_new$pxl_row_in_fullres_original
            )
            expect_identical(
                spatialCoords(spe_small)[, "pxl_col_in_fullres"],
                spe_new$pxl_col_in_fullres_original
            )

            #   Euclidean error for the Euclidean algorithm must be between 0
            #   and (less than) 1 spot; for the LSAP algorithm, it can
            #   theoretically be higher, but shouldn't be under reasonable
            #   circumstances
            expect_equal(
                all(
                    (spe_new$euclidean_error >= 0) &
                        (spe_new$euclidean_error < 1)
                ),
                TRUE
            )

            #   Shared neighbors must similarly be between 0 and 1 (it's a
            #   proportion). NAs are allowed when there are no neighbors
            #   originally
            temp <- spe_new$shared_neighbors[!is.na(spe_new$shared_neighbors)]
            expect_equal(all((temp >= 0) & (temp <= 1)), TRUE)
        }
    }
)
