#' Identify breakpoints in a \code{marge} model.
#'
#' @name extractBreakpoints
#' @author Jack R. Leary
#' @description Extracts the breakpoints from a fitted \code{marge} model. 
#' Note - this function relies on the name of the pseudotime variable not 
#' having any numeric characters in it e.g., "pseudotime" or "PT" would be 
#' fine but "pseudotime1" would not. If multiple lineages exist, use letters 
#' to denote lineages instead of numbers e.g., "Lineage_A" and "Lineage_B". 
#' This is currently handled automatically in \code{\link{testDynamic}}, so 
#' don't change anything.
#' @param model The \code{marge} model to analyze. Defaults to NULL.
#' @param directions Should the directions of the hinge functions also be 
#' extracted? Defaults to TRUE.
#' @return A data.frame of breakpoints & their directions.
#' @export
#' @examples
#' data(sim_counts)
#' data(sim_pseudotime)
#' cell_offset <- createCellOffset(sim_counts)
#' marge_model <- marge2(sim_pseudotime,
#'     Y = BiocGenerics::counts(sim_counts)[4, ],
#'     Y.offset = cell_offset
#' )
#' breakpoint_df <- extractBreakpoints(model = marge_model)
extractBreakpoints <- function(model = NULL, directions = TRUE) {
    # check inputs
    if (is.null(model)) {
        stop("Model input is missing from extractBreakpoints().")
    }
    if (!inherits(model, "marge")) {
        stop("Model must be of class marge.")
    }
    # find changepoints -> need to parse them out of coefficient
    # names generated by marge2()
    coef_names <- model$coef_names
    coef_names <- coef_names[-which(coef_names == "Intercept")]
    coef_nums <- gsub("[A-z]|[a-z]", "", coef_names)
    brkpt_df <- data.frame(Breakpoint = as.numeric(coef_nums))
    if (directions) {
        brkpt_dirs <- ifelse(grepl("h_[a-z].*_[0-9]",
            coef_names,
            ignore.case = TRUE
        ),
        "Right",
        "Left"
        )
        brkpt_df$Direction <- brkpt_dirs
    }
    return(brkpt_df)
}
