% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metagene_functions.R
\name{get_metagene}
\alias{get_metagene}
\title{Retrieves the metagene data from a .ribo file}
\usage{
get_metagene(
  ribo.object,
  site,
  range.lower = length_min(ribo.object),
  range.upper = length_max(ribo.object),
  transcript = TRUE,
  length = TRUE,
  alias = FALSE,
  compact = TRUE,
  experiment = experiments(ribo.object)
)
}
\arguments{
\item{ribo.object}{A 'Ribo' object}

\item{site}{"start" or "stop" site coverage}

\item{range.lower}{Lower bound of the read length, inclusive}

\item{range.upper}{Upper bound of the read length, inclusive}

\item{transcript}{Logical value that denotes if the metagene information should be summed across transcripts}

\item{length}{Logical value that denotes if the metagene information should be summed across read lengths}

\item{alias}{Option to report the transcripts as aliases/nicknames}

\item{compact}{Option to return a DataFrame with Rle and factor as opposed to a raw data.frame}

\item{experiment}{List of experiment names}
}
\value{
An annotated DataFrame or data.frame (if the compact parameter is set to FALSE) of the
metagene information for either the 'stop' or 'start' site provided in the 'site' parameter. The
returned data frame will have a length column when the 'length' parameter is set to FALSE, indicating
the returned data frame will have a transcript column whe the 'transcript' parameter is set to FALSE,
indicating that the count information will not be summed across the transcripts.
In the case that transcript parameter is 'FALSE', the returned data frame will present the transcripts according
to the aliases specified at the creation of the ribo object if the 'alias' parameter is set to TRUE.
}
\description{
The function \code{\link{get_metagene}} returns a data frame that provides
the coverage at the positions surrounding the metagene start or stop site.
}
\details{
The dimensions of the returned data frame depend on the parameters
range.lower, range.upper, length, and transcript.

The param 'length' condenses the read lengths together.
When length is TRUE and transcript is FALSE, the
data frame presents information for each transcript across
all of the read lengths. That is, each transcript has a value
that is the sum of all of the counts across every read length.
As a result, information about the transcript at each specific
read length is lost.

The param 'transcripts' condenses the transcripts together.
When transcript is TRUE and length is FALSE, the data
frame presents information at each read length between range.lower and
range.upper inclusive. That is, each separate read length denotes the
sum of counts from every transcript. As a result, information about the
counts of each individual transcript is lost.

If both 'length' and 'transcript' are TRUE, then the resulting
data frame prints out one row for each experiment. This provides the metagene
information across all transcripts and all reads in a given experiment.

If both length' and 'transcript' are FALSE, no calculations are done to the data,
all information is preserved for both the read length and the transcript.
The data frame would just present the entire stored raw data
from the read length 'range.lower' to the read length 'range.upper' which in most
cases would result in a slow run time with a massive DataFrame returned.

When 'transcript' is set to FALSE, the 'alias' parameter specifies whether
or not the returned DataFrame should present each transcript as an alias
instead of the original name. If 'alias' is set to TRUE, then the returned
data frame will contain the aliases rather than the original
reference names of the .ribo file.
}
\examples{

#generate the ribo object by providing the file.path to the ribo file
file.path <- system.file("extdata", "sample.ribo", package = "ribor")
sample <- Ribo(file.path)


#extract the total metagene information for all experiments
#across the read lengths and transcripts of the start site
#from read length 2 to 5
metagene_info <- get_metagene(ribo.object = sample,
                              site = "start",
                              range.lower = 2,
                              range.upper = 5,
                              length = TRUE,
                              transcript = TRUE,
                              experiment = experiments(sample))


#Note that length, transcript, and experiments in this case are the
#default values and can be left out. The following generates the same output.

metagene_info <- get_metagene(ribo.object = sample,
                              site = "start",
                              range.lower = 2,
                              range.upper = 5)

}
\seealso{
\code{\link{Ribo}} to generate the necessary 'Ribo' class object,
\code{\link{plot_metagene}} to visualize the metagene data,
\code{\link{get_tidy_metagene}} to obtain tidy metagene data under certain conditions
}
