% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DESeq2Report.R
\name{DESeq2Report}
\alias{DESeq2Report}
\title{Generate a HTML/PDF report exploring DESeq2 results}
\usage{
DESeq2Report(
  dds,
  project = "",
  intgroup,
  colors = NULL,
  res = NULL,
  nBest = 500,
  nBestFeatures = 20,
  customCode = NULL,
  outdir = "DESeq2Exploration",
  output = "DESeq2Exploration",
  browse = interactive(),
  device = "png",
  template = NULL,
  searchURL = "http://www.ncbi.nlm.nih.gov/gene/?term=",
  theme = NULL,
  digits = 2,
  ...
)
}
\arguments{
\item{dds}{A \link[DESeq2]{DESeqDataSet} object with the results from
running \link[DESeq2]{DESeq}.}

\item{project}{The title of the project.}

\item{intgroup}{interesting groups: a character vector of names in
\code{colData(x)} to use for grouping. This parameter is passed to functions
such as \link[DESeq2]{plotPCA}.}

\item{colors}{vector of colors used in heatmap. If \code{NULL}, then a
a default set of colors will be used. This argument is passed to
\link[pheatmap]{pheatmap}.}

\item{res}{A \link[DESeq2]{DESeqResults} object. If \code{NULL}, then
\link[DESeq2]{results} will be used on \code{dds} with default parameters.}

\item{nBest}{The number of features to include in the interactive
table. Features are ordered by their adjusted p-values.}

\item{nBestFeatures}{The number of best features to make plots of their
counts. We recommend a small number, say 20.}

\item{customCode}{An absolute path to a child R Markdown file with code to be
evaluated before the reproducibility section. Its useful for users who want
to customize the report by adding conclusions derived from the data and/or
further quality checks and plots.}

\item{outdir}{The name of output directory.}

\item{output}{The name of output HTML file (without the html extension).}

\item{browse}{If \code{TRUE} the HTML report is opened in your browser once
it's completed.}

\item{device}{The graphical device used when knitting. See more at
\url{http://yihui.name/knitr/options} (\code{dev} argument).}

\item{template}{Template file to use for the report. If not provided, will
use the default file found in DESeq2Exploration/DESeq2Exploration.Rmd
within the package source.}

\item{searchURL}{A url used for searching the name of the features in
the web. By default \code{http://www.ncbi.nlm.nih.gov/gene/?term=} is used
which is the recommended option when features are genes. It's only used
when the output is a HTML file.}

\item{theme}{A ggplot2 \link[ggplot2]{theme} to use for the plots made with
ggplot2.}

\item{digits}{The number of digits to round to in the interactive table of
the top \code{nBestFeatures}. Note that p-values and adjusted p-values won't
be rounded.}

\item{...}{Arguments passed to other methods and/or advanced arguments.
Advanced arguments:
\describe{
\item{software }{ The name of the package used for performing the
differential expression analysis. Either \code{DESeq2} or \code{edgeR}.}
\item{dge }{ A \link[edgeR]{DGEList} object. \code{NULL} by default and only
used by \link{edgeReport}.}
\item{theCall }{ The function call. \code{NULL} by default and only used by
\link{edgeReport}.}
\item{output_format }{ Either \code{html_document}, \code{pdf_document} or
\code{knitrBootstrap::bootstrap_document} unless you modify the YAML
template.}
\item{clean }{ Logical, whether to clean the results or not. Passed to
\link[rmarkdown]{render}.}
}}
}
\value{
An HTML report with a basic exploration for the given set of DESeq2
results. See an example at \url{http://leekgroup.github.io/regionReport/reference/DESeq2Report-example/DESeq2Exploration.html}.
}
\description{
This function generates a HTML report with exploratory data analysis plots
for DESeq2 results created with \link[DESeq2]{DESeq}. Other output formats
are possible such as PDF but lose the interactivity. Users can easily append
to the report by providing a R Markdown file to \code{customCode}, or can
customize the entire template by providing an R Markdown file to
\code{template}.
}
\details{
Set \code{output_format} to \code{'knitrBootstrap::bootstrap_document'} or
\code{'pdf_document'} if you want a HTML report styled by knitrBootstrap or
a PDF report respectively. If using knitrBootstrap, we recommend the version
available only via GitHub at \url{https://github.com/jimhester/knitrBootstrap}
which has nicer features than the current version available via CRAN. You can
also set the \code{output_format} to \code{'html_document'} for a HTML
report styled by rmarkdown. The default is set to
\code{'BiocStyle::html_document'}.

If you modify the YAML front matter of \code{template}, you can use other
values for \code{output_format}.

The HTML report styled with knitrBootstrap can be smaller in size than the
\code{'html_document'} report.
}
\examples{

## Load example data from the pasilla package as done in the DESeq2 vignette
## at
## <https://bioconductor.org/packages/release/bioc/vignettes/DESeq2/inst/doc/DESeq2.html#count-matrix-input>.
library("pasilla")
pasCts <- system.file("extdata",
    "pasilla_gene_counts.tsv",
    package = "pasilla", mustWork = TRUE
)
pasAnno <- system.file("extdata",
    "pasilla_sample_annotation.csv",
    package = "pasilla", mustWork = TRUE
)
cts <- as.matrix(read.csv(pasCts, sep = "\t", row.names = "gene_id"))
coldata <- read.csv(pasAnno, row.names = 1)
coldata <- coldata[, c("condition", "type")]
coldata$condition <- factor(coldata$condition)
coldata$type <- factor(coldata$type)
rownames(coldata) <- sub("fb", "", rownames(coldata))
cts <- cts[, rownames(coldata)]

## Create DESeqDataSet object from the pasilla package
library("DESeq2")
dds <- DESeqDataSetFromMatrix(
    countData = cts,
    colData = coldata,
    design = ~condition
)
dds <- DESeq(dds)

## The output will be saved in the 'DESeq2Report-example' directory
dir.create("DESeq2Report-example", showWarnings = FALSE, recursive = TRUE)

## Generate the HTML report
report <- DESeq2Report(dds, "DESeq2-example", c("condition", "type"),
    outdir = "DESeq2Report-example"
)

if (interactive()) {
    ## Browse the report
    browseURL(report)
}

## See the example output at
## http://leekgroup.github.io/regionReport/reference/DESeq2Report-example/DESeq2Exploration.html
\dontrun{
## Note that you can run the example using:
example("DESeq2Report", "regionReport", ask = FALSE)
}

}
\author{
Leonardo Collado-Torres
}
