test_that("digestFastqs works as expected for primer experiments", {
  fqt1 <- system.file("extdata/leujunt0_1.fastq.gz", package = "mutscan")
  fqt2 <- system.file("extdata/leujunt0_2.fastq.gz", package = "mutscan")
  leu <- c(ATF2 = "GATCCTGATGAAAAAAGGAGAAAGTTTTTAGAGCGAAATAGAGCAGCAGCTTCAAGATGCCGACAAAAAAGGAAAGTCTGGGTTCAGTCTTTAGAGAAGAAAGCTGAAGACTTGAGTTCATTAAATGGTCAGCTGCAGAGTGAAGTCACCCTGCTGAGAAATGAAGTGGCACAGCTGAAACAGCTTCTTCTGGCT",
           ATF7 = "GATCCAGATGAGCGACGGCAGCGCTTTCTGGAGCGCAACCGGGCTGCAGCCTCCCGCTGCCGCCAAAAGCGAAAGCTGTGGGTGTCCTCCCTAGAGAAGAAGGCCGAAGAACTCACTTCTCAGAACATTCAGCTGAGTAATGAAGTCACATTACTACGCAATGAGGTGGCCCAGTTGAAACAGCTACTGTTAGCT",
           CREB5 = "GATCCGGACGAGAGGCGGCGGAAATTTCTGGAACGGAACCGGGCAGCTGCCACCCGCTGCAGACAGAAGAGGAAGGTCTGGGTGATGTCATTGGAAAAGAAAGCAGAAGAACTCACCCAGACAAACATGCAGCTTCAGAATGAAGTGTCTATGTTGAAAAATGAGGTGGCCCAGCTGAAACAGTTGTTGTTAACA",
           ATF3 = "GAAGAAGATGAAAGGAAAAAGAGGCGACGAGAAAGAAATAAGATTGCAGCTGCAAAGTGCCGAAACAAGAAGAAGGAGAAGACGGAGTGCCTGCAGAAAGAGTCGGAGAAGCTGGAAAGTGTGAATGCTGAACTGAAGGCTCAGATTGAGGAGCTCAAGAACGAGAAGCAGCATTTGATATACATGCTCAACCTT",
           JDP2 = "GAGGAAGAGGAGCGAAGGAAAAGGCGCCGGGAGAAGAACAAAGTCGCAGCAGCCCGATGCCGGAACAAGAAGAAGGAGCGCACGGAGTTTCTGCAGCGGGAATCCGAGCGGCTGGAACTCATGAACGCAGAGCTGAAGACCCAGATTGAGGAGCTGAAGCAGGAGCGGCAGCAGCTCATCCTGATGCTGAACCGA",
           ATF4 = "GAGAAACTGGATAAGAAGCTGAAAAAAATGGAGCAAAACAAGACAGCAGCCACTAGGTACCGCCAGAAGAAGAGGGCGGAGCAGGAGGCTCTTACTGGTGAGTGCAAAGAGCTGGAAAAGAAGAACGAGGCTCTAAAAGAGAGGGCGGATTCCCTGGCCAAGGAGATCCAGTACCTGAAAGATTTGATAGAAGAG",
           ATF5 = "ACCCGAGGGGACCGCAAGCAAAAGAAGAGAGACCAGAACAAGTCGGCGGCTCTGAGGTACCGCCAGCGGAAGCGGGCAGAGGGTGAGGCCCTGGAGGGCGAGTGCCAGGGGCTGGAGGCACGGAATCGCGAGCTGAAGGAACGGGCAGAGTCCGTGGAGCGCGAGATCCAGTACGTCAAGGACCTGCTCATCGAG",
           CREBZF = "AGTCCCCGGAAGGCGGCGGCGGCCGCTGCCCGCCTTAATCGACTGAAGAAGAAGGAGTACGTGATGGGGCTGGAGAGTCGAGTCCGGGGTCTGGCAGCCGAGAACCAGGAGCTGCGGGCCGAGAATCGGGAGCTGGGCAAACGCGTACAGGCACTGCAGGAGGAGAGTCGCTACCTACGGGCAGTCTTAGCCAAC",
           BATF2 = "CCCAAGGAGCAACAAAGGCAGCTGAAGAAGCAGAAGAACCGGGCAGCCGCCCAGCGAAGCCGGCAGAAGCACACAGACAAGGCAGACGCCCTGCACCAGCAGCACGAGTCTCTGGAAAAAGACAACCTCGCCCTGCGGAAGGAGATCCAGTCCCTGCAGGCCGAGCTGGCGTGGTGGAGCCGGACCCTGCACGTG",
           BATF3 = "GAGGATGATGACAGGAAGGTCCGAAGGAGAGAAAAAAACCGAGTTGCTGCTCAGAGAAGTCGGAAGAAGCAGACCCAGAAGGCTGACAAGCTCCATGAGGAATATGAGAGCCTGGAGCAAGAAAACACCATGCTGCGGAGAGAGATCGGGAAGCTGACAGAGGAGCTGAAGCACCTGACAGAGGCACTGAAGGAG",
           CEBPE = "AAAGATAGCCTTGAGTACCGGCTGAGGCGGGAGCGCAACAACATCGCCGTGCGCAAGAGCCGAGACAAGGCCAAGAGGCGCATTCTGGAGACGCAGCAGAAGGTGCTGGAGTACATGGCAGAGAACGAGCGCCTCCGCAGCCGCGTGGAGCAGCTCACCCAGGAGCTAGACACCCTCCGCAACCTCTTCCGCCAG",
           BACH1 = "CTGGATTGTATCCATGATATTCGAAGAAGAAGTAAAAACAGAATTGCTGCACAGCGCTGTCGCAAGAGAAAACTTGACTGTATACAGAATCTTGAATCAGAAATTGAGAAGCTGCAAAGTGAAAAGGAGAGCTTGTTGAAGGAAAGAGATCACATTTTGTCAACTCTGGGTGAGACAAAGCAGAACCTAACTGGA",
           BACH2 = "TTAGAGTTTATTCATGATGTCCGACGGCGCAGCAAGAACCGCATCGCGGCCCAGCGCTGCCGCAAAAGGAAACTGGACTGTATTCAGAATTTAGAATGTGAAATCCGCAAATTGGTGTGTGAGAAAGAGAAACTGTTGTCAGAGAGGAATCAACTGAAAGCATGCATGGGGGAACTGTTGGACAACTTCTCCTGC",
           NFE2L1 = "CTGAGCCTCATCCGAGACATCCGGCGCCGGGGCAAGAACAAGATGGCGGCGCAGAACTGCCGCAAGCGCAAGCTGGACACCATCCTGAATCTGGAGCGTGATGTGGAGGACCTGCAGCGTGACAAAGCCCGGCTGCTGCGGGAGAAAGTGGAGTTCCTGCGCTCCCTGCGACAGATGAAGCAGAAGGTCCAGAGC",
           NFE2 = "CTAGCGCTAGTCCGGGACATCCGACGACGGGGCAAAAACAAGGTGGCAGCCCAGAACTGCCGCAAGAGGAAGCTGGAAACCATTGTGCAGCTGGAGCGGGAGCTGGAGCGGCTGACCAATGAACGGGAGCGGCTTCTCAGGGCCCGCGGGGAGGCAGACCGGACCCTGGAGGTCATGCGCCAACAGCTGACAGAG",
           NFIL3 = "AAGAAAGATGCTATGTATTGGGAAAAAAGGCGGAAAAATAATGAAGCTGCCAAAAGATCTCGTGAGAAGCGTCGACTGAATGACCTGGTTTTAGAGAACAAACTAATTGCACTGGGAGAAGAAAACGCCACTTTAAAAGCTGAGCTGCTTTCACTAAAATTAAAGTTTGGTTTAATTAGCTCCACAGCATATGCT",
           FOS = "GAAGAAGAAGAGAAAAGGAGAATCCGAAGGGAAAGGAATAAGATGGCTGCAGCCAAATGCCGCAACCGGAGGAGGGAGCTGACTGATACACTCCAAGCGGAGACAGACCAACTAGAAGATGAGAAGTCTGCTTTGCAGACCGAGATTGCCAACCTGCTGAAGGAGAAGGAAAAACTAGAGTTCATCCTGGCAGCT",
           FOSB = "GAGGAAGAGGAGAAGCGAAGGGTGCGCCGGGAACGAAATAAACTAGCAGCAGCTAAATGCAGGAACCGGCGGAGGGAGCTGACCGACCGACTCCAGGCGGAGACAGATCAGTTGGAGGAAGAAAAAGCAGAGCTGGAGTCGGAGATCGCCGAGCTCCAAAAGGAGAAGGAACGTCTGGAGTTTGTGCTGGTGGCC",
           FOSL1 = "GAGGAAGAGGAGCGCCGCCGAGTAAGGCGCGAGCGGAACAAGCTGGCTGCGGCCAAGTGCAGGAACCGGAGGAAGGAACTGACCGACTTCCTGCAGGCGGAGACTGACAAACTGGAAGATGAGAAATCTGGGCTGCAGCGAGAGATTGAGGAGCTGCAGAAGCAGAAGGAGCGCCTAGAGCTGGTGCTGGAAGCC",
           FOSL2 = "GAAGAGGAGGAGAAGCGTCGCATCCGGCGGGAGAGGAACAAGCTGGCTGCAGCCAAGTGCCGGAACCGACGCCGGGAGCTGACAGAGAAGCTGCAGGCGGAGACAGAGGAGCTGGAGGAGGAGAAGTCAGGCCTGCAGAAGGAGATTGCTGAGCTGCAGAAGGAGAAGGAGAAGCTGGAGTTCATGTTGGTGGCT",
           MAFB = "GTGATCCGCCTGAAGCAGAAGCGGCGGACCCTGAAGAACCGGGGCTACGCCCAGTCTTGCAGGTATAAACGCGTCCAGCAGAAGCACCACCTGGAGAATGAGAAGACGCAGCTCATTCAGCAGGTGGAGCAGCTTAAGCAGGAGGTGTCCCGGCTGGCCCGCGAGAGAGACGCCTACAAGGTCAAGTGCGAGAAA",
           JUN = "CAGGAGCGGATCAAGGCGGAGAGGAAGCGCATGAGGAACCGCATCGCTGCCTCCAAGTGCCGAAAAAGGAAGCTGGAGAGAATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTTAAACAGAAAGTCATGAAC",
           JUNB = "CAAGAGCGCATCAAAGTGGAGCGCAAGCGGCTGCGGAACCGGCTGGCGGCCACCAAGTGCCGGAAGCGGAAGCTGGAGCGCATCGCGCGCCTGGAGGACAAGGTGAAGACGCTCAAGGCCGAGAACGCGGGGCTGTCGAGTACCGCCGGCCTCCTCCGGGAGCAGGTGGCCCAGCTCAAACAGAAGGTCATGACC",
           JUND = "CAGGAGCGCATCAAGGCGGAGCGCAAGCGGCTGCGCAACCGCATCGCCGCCTCCAAGTGCCGCAAGCGCAAGCTGGAGCGCATCTCGCGCCTGGAAGAGAAAGTGAAGACCCTCAAGAGTCAGAACACGGAGCTGGCGTCCACGGCGAGCCTGCTGCGCGAGCAGGTGGCGCAGCTCAAGCAGAAAGTCCTCAGC",
           CREB3 = "GAACAAATTCTGAAACGTGTGCGGAGGAAGATTCGAAATAAAAGATCTGCTCAAGAGAGCCGCAGGAAAAAGAAGGTGTATGTTGGGGGTTTAGAGAGCAGGGTCTTGAAATACACAGCCCAGAATATGGAGCTTCAGAACAAAGTACAGCTTCTGGAGGAACAGAATTTGTCCCTTCTAGATCAACTGAGGAAA",
           HLF = "CTGAAGGATGACAAGTACTGGGCAAGGCGCAGAAAGAACAACATGGCAGCCAAGCGCTCCCGCGACGCCCGGAGGCTGAAAGAGAACCAGATCGCCATCCGGGCCTCGTTCCTGGAGAAGGAGAACTCGGCCCTCCGCCAGGAGGTGGCTGACTTGAGGAAGGAGCTGGGCAAATGCAAGAACATACTTGCCAAG",
           MAFG = "ATCGTCCAGCTGAAGCAGCGCCGGCGCACGCTCAAGAACCGCGGCTACGCTGCCAGCTGCCGCGTGAAGCGGGTGACGCAGAAGGAGGAGCTGGAGAAGCAGAAGGCGGAGCTGCAGCAGGAGGTGGAGAAGCTGGCCTCAGAGAACGCCAGCATGAAGCTGGAGCTCGACGCGCTGCGCTCCAAGTACGAGGCG",
           MAFK = "GTGACCCGCCTGAAGCAGCGTCGGCGCACACTCAAGAACCGCGGCTACGCGGCCAGCTGCCGCATCAAGCGGGTGACGCAGAAGGAGGAGCTGGAGCGGCAGCGCGTGGAGCTGCAGCAGGAGGTGGAGAAGCTGGCGCGTGAGAACAGCAGCATGCGGCTGGAGCTGGACGCCCTGCGCTCCAAGTACGAGGCG",
           XBP1 = "AGCCCCGAGGAGAAGGCGCTGAGGAGGAAACTGAAAAACAGAGTAGCAGCTCAGACTGCCAGAGATCGAAAGAAGGCTCGAATGAGTGAGCTGGAACAGCAAGTGGTAGATTTAGAAGAAGAGAACCAAAAACTTTTGCTAGAAAATCAGCTTTTACGAGAGAAAACTCATGGCCTTGTAGTTGAGAACCAGGAG",
           ATF6 = "ATTGCTGTGCTAAGGAGACAGCAACGTATGATAAAAAATCGAGAATCCGCTTGTCAGTCTCGCAAGAAGAAGAAAGAATATATGCTAGGGTTAGAGGCGAGATTAAAGGCTGCCCTCTCAGAAAACGAGCAACTGAAGAAAGAAAATGGAACACTGAAGCGGCAGCTGGATGAAGTTGTGTCAGAGAACCAGAGG",
           ATF6B = "GCAAAGCTGCTGAAGCGGCAGCAGCGAATGATCAAGAACCGGGAGTCAGCCTGCCAGTCCCGGAGAAAGAAGAAAGAGTATCTGCAGGGACTGGAGGCTCGGCTGCAAGCAGTACTGGCTGACAACCAGCAGCTCCGCCGAGAGAATGCTGCCCTCCGGCGGCGGCTGGAGGCCCTGCTGGCTGAAAACAGCGAG",
           CEBPA = "AAGAACAGCAACGAGTACCGGGTGCGGCGCGAGCGCAACAACATCGCGGTGCGCAAGAGCCGCGACAAGGCCAAGCAGCGCAACGTGGAGACGCAGCAGAAGGTGCTGGAGCTGACCAGTGACAATGACCGCCTGCGCAAGCGGGTGGAACAGCTGAGCCGCGAACTGGACACGCTGCGGGGCATCTTCCGCCAG",
           CEBPB = "AAGCACAGCGACGAGTACAAGATCCGGCGCGAGCGCAACAACATCGCCGTGCGCAAGAGCCGCGACAAGGCCAAGATGCGCAACCTGGAGACGCAGCACAAGGTCCTGGAGCTCACGGCCGAGAACGAGCGGCTGCAGAAGAAGGTGGAGCAGCTGTCGCGCGAGCTCAGCACCCTGCGGAACTTGTTCAAGCAG",
           CEBPD = "CGCGGCAGCCCCGAGTACCGGCAGCGGCGCGAGCGCAACAACATCGCCGTGCGCAAGAGCCGCGACAAGGCCAAGCGGCGCAACCAGGAGATGCAGCAGAAGTTGGTGGAGCTGTCGGCTGAGAACGAGAAGCTGCACCAGCGCGTGGAGCAGCTCACGCGGGACCTGGCCGGCCTCCGGCAGTTCTTCAAGCAG",
           CEBPG = "CGAAACAGTGACGAGTATCGGCAACGCCGAGAGAGGAACAACATGGCTGTGAAAAAGAGCCGGTTGAAAAGCAAGCAGAAAGCACAAGACACACTGCAGAGAGTCAATCAGCTCAAAGAAGAGAATGAACGGTTGGAAGCAAAAATCAAATTGCTGACCAAGGAATTAAGTGTACTCAAAGATTTGTTTCTTGAG",
           CREB1 = "GAAGCAGCACGAAAGAGAGAGGTCCGTCTAATGAAGAACAGGGAAGCAGCTCGAGAGTGTCGTAGAAAGAAGAAAGAATATGTGAAATGTTTAGAAAACAGAGTGGCAGTGCTTGAAAATCAAAACAAGACATTGATTGAGGAGCTAAAAGCACTTAAGGACCTTTACTGCCACAAATCAGAT",
           CREB3L1 = "GAGAAGGCCTTGAAGAGAGTCCGGAGGAAAATCAAGAACAAGATCTCAGCCCAGGAGAGCCGTCGTAAGAAGAAGGAGTATGTGGAGTGTCTAGAAAAGAAGGTGGAGACATTTACATCTGAGAACAATGAACTGTGGAAGAAGGTGGAGACCCTGGAGAATGCCAACAGGACCCTGCTCCAGCAGCTGCAGAAA",
           CREB3L2 = "GAGAAGGCCCTGAAGAAAATTCGGAGGAAGATCAAGAATAAGATTTCTGCTCAGGAAAGTAGGAGAAAGAAGAAAGAATACATGGACAGCCTGGAGAAAAAAGTGGAGTCTTGTTCAACTGAGAACTTGGAGCTTCGGAAGAAGGTAGAGGTTCTAGAGAACACTAATAGGACTCTCCTTCAGCAACTCCAGAAG",
           CREB3L3 = "GAGCGAGTGCTGAAAAAAATCCGCCGGAAAATCCGGAACAAGCAGTCGGCGCAAGAAAGCAGGAAGAAGAAGAAGGAATATATCGATGGCCTGGAGACTCGGATGTCAGCTTGCACTGCTCAGAATCAGGAGTTACAGAGGAAAGTCTTGCATCTCGAGAAGCAAAACCTGTCCCTCTTGGAGCAACTGAAGAAA",
           CREB3L4 = "GAGAGGGTCCTCAAGAAGGTCAGGAGGAAAATCCGTAACAAGCAGTCAGCTCAGGACAGTCGGCGGCGGAAGAAGGAGTACATTGATGGGCTGGAGAGCAGGGTGGCAGCCTGTTCTGCACAGAACCAAGAATTACAGAAAAAAGTCCAGGAGCTGGAGAGGCACAACATCTCCTTGGTAGCTCAGCTCCGCCAG",
           CREBL2 = "CCAGCCAAAATTGACTTGAAAGCAAAACTTGAGAGGAGCCGGCAGAGTGCAAGAGAATGCCGAGCCCGAAAAAAGCTGAGATATCAGTATTTGGAAGAGTTGGTATCCAGTCGAGAAAGAGCTATATGTGCCCTCAGAGAGGAACTGGAAATGTACAAGCAGTGGTGCATGGCAATGGACCAAGGAAAAATCCCT",
           CREBRF = "CCCTTAACAGCCCGACCAAGGTCAAGGAAGGAAAAAAATAAGCTGGCTTCCAGAGCTTGTCGGTTAAAGAAGAAAGCCCAGTATGAAGCTAATAAAGTGAAATTATGGGGCCTCAACACAGAATATGATAATTTATTGTTTGTAATCAACTCCATCAAGCAAGAGATTGTAAACCGGGTACAGAATCCAAGAGAT",
           DBP = "CAGAAGGATGAGAAATACTGGAGCCGGCGGTACAAGAACAACGAGGCAGCCAAGCGGTCCCGTGACGCCCGGCGGCTCAAGGAGAACCAGATATCGGTGCGGGCGGCCTTCCTGGAGAAGGAGAACGCCCTGCTGCGGCAGGAAGTTGTGGCCGTGCGCCAGGAGCTGTCCCACTACCGCGCCGTGCTGTCCCGA",
           NFE2L2 = "CTTGCATTAATTCGGGATATACGTAGGAGGGGTAAGAATAAAGTGGCTGCTCAGAATTGCAGAAAAAGAAAACTGGAAAATATAGTAGAACTAGAGCAAGATTTAGATCATTTGAAAGATGAAAAAGAAAAATTGCTCAAAGAAAAAGGAGAAAATGACAAAAGCCTTCACCTACTGAAAAAACAACTCAGCACC",
           NFE2L3 = "GTCTCACTTATCCGTGACATCAGACGAAGAGGGAAAAATAAAGTTGCTGCGCAGAACTGTCGTAAACGCAAATTGGACATAATTTTGAATTTAGAAGATGATGTATGTAACTTGCAAGCAAAGAAGGAAACTCTTAAGAGAGAGCAAGCACAATGTAACAAAGCTATTAACATAATGAAACAGAAACTGCATGAC",
           TEF = "CAGAAGGATGAAAAGTACTGGACAAGACGCAAGAAGAACAACGTGGCAGCTAAACGGTCACGGGATGCCCGGCGCCTGAAAGAGAATCAGATCACCATCCGGGCAGCCTTCCTGGAGAAGGAGAACACAGCCCTGCGGACGGAGGTGGCCGAGCTACGCAAGGAGGTGGGCAAGTGCAAGACCATCGTGTCCAAG"
  )
  ## default arguments
  Ldef <- list(
    fastqForward = fqt1, fastqReverse = fqt2, 
    mergeForwardReverse = FALSE, 
    minOverlap = 0, maxOverlap = 0, maxFracMismatchOverlap = 0, greedyOverlap = TRUE, 
    revComplForward = FALSE, revComplReverse = FALSE,
    elementsForward = "SPV", elementsReverse = "SPVS",
    elementLengthsForward = c(-1, -1, -1),
    elementLengthsReverse = c(-1, -1, 96, -1),
    adapterForward = "", 
    adapterReverse = "",
    primerForward = "GTCAGGTGGAGGCGGATCC",
    primerReverse = "GAAAAAGGAAGCTGGAGAGA",
    wildTypeForward = leu,
    wildTypeReverse = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTT", 
    constantForward = "", 
    constantReverse = "", 
    avePhredMinForward = 20.0, avePhredMinReverse = 20.0,
    variableNMaxForward = 0, variableNMaxReverse = 0, 
    umiNMax = 0,
    nbrMutatedCodonsMaxForward = 0,
    nbrMutatedCodonsMaxReverse = 1,
    nbrMutatedBasesMaxForward = -1,
    nbrMutatedBasesMaxReverse = -1,
    forbiddenMutatedCodonsForward = "",
    forbiddenMutatedCodonsReverse = "NNW",
    useTreeWTmatch = FALSE,
    mutatedPhredMinForward = 0.0, mutatedPhredMinReverse = 0.0,
    mutNameDelimiter = ".", 
    constantMaxDistForward = -1,
    constantMaxDistReverse = -1,
    umiCollapseMaxDist = 0,
    filteredReadsFastqForward = "",
    filteredReadsFastqReverse = "",
    maxNReads = -1, verbose = FALSE,
    nThreads = 1, chunkSize = 1000, 
    maxReadLength = 1024
  )
  
  res <- do.call(digestFastqs, Ldef)
  
  expect_equal(res$filterSummary$nbrTotal, 1000L)
  expect_equal(res$filterSummary$f1_nbrAdapter, 0L)
  expect_equal(res$filterSummary$f2_nbrNoPrimer, 126L)
  expect_equal(res$filterSummary$f3_nbrReadWrongLength, 0L)
  expect_equal(res$filterSummary$f4_nbrNoValidOverlap, 0L)
  expect_equal(res$filterSummary$f5_nbrAvgVarQualTooLow, 0L)
  expect_equal(res$filterSummary$f6_nbrTooManyNinVar, 76L)
  expect_equal(res$filterSummary$f7_nbrTooManyNinUMI, 0L)
  expect_equal(res$filterSummary$f8_nbrTooManyBestWTHits, 0L)
  expect_equal(res$filterSummary$f9_nbrMutQualTooLow, 0L)
  expect_equal(res$filterSummary$f10a_nbrTooManyMutCodons, 58L + 137L)
  expect_equal(res$filterSummary$f10b_nbrTooManyMutBases, 0L)
  expect_equal(res$filterSummary$f11_nbrForbiddenCodons, 3L)
  expect_equal(res$filterSummary$f12_nbrTooManyMutConstant, 0L)
  expect_equal(res$filterSummary$f13_nbrTooManyBestConstantHits, 0L)
  expect_equal(res$filterSummary$nbrRetained, 600L)
  
  for (nm in setdiff(names(Ldef), c("forbiddenMutatedCodonsForward", "forbiddenMutatedCodonsReverse", "verbose", "fastqForward", "fastqReverse"))) {
    expect_equal(res$parameters[[nm]], Ldef[[nm]], ignore_attr = TRUE)
  }
  for (nm in c("fastqForward", "fastqReverse")) {
    expect_equal(res$parameters[[nm]], normalizePath(Ldef[[nm]], mustWork = FALSE), 
                 ignore_attr = TRUE)
  }
  
  expect_equal(sum(res$summaryTable$nbrReads), res$filterSummary$nbrRetained)
  expect_equal(sum(res$summaryTable$nbrReads == 2), 11L)
  expect_equal(sort(res$summaryTable$mutantName[res$summaryTable$nbrReads == 2]),
               sort(c("BACH2.0.WT_r.6.CCC", "BATF2.0.WT_r.21.TGG", "BATF2.0.WT_r.7.GGG", "CEBPB.0.WT_r.12.CTC",
                      "CEBPD.0.WT_r.13.CCC", "CEBPD.0.WT_r.23.GAG", "CREB3L3.0.WT_r.22.GGG",
                      "FOSL1.0.WT_r.22.CCC", "FOSL2.0.WT_r.13.GGG", "JUNB.0.WT_r.0.WT", "XBP1.0.WT_r.0.WT")))
  expect_equal(sort(res$summaryTable$mutantNameCodon[res$summaryTable$nbrReads == 2]),
               sort(c("BACH2.0.WT_r.6.CCC", "BATF2.0.WT_r.21.TGG", "BATF2.0.WT_r.7.GGG", "CEBPB.0.WT_r.12.CTC",
                      "CEBPD.0.WT_r.13.CCC", "CEBPD.0.WT_r.23.GAG", "CREB3L3.0.WT_r.22.GGG",
                      "FOSL1.0.WT_r.22.CCC", "FOSL2.0.WT_r.13.GGG", "JUNB.0.WT_r.0.WT", "XBP1.0.WT_r.0.WT")))
  expect_equal(sort(res$summaryTable$mutantNameBase[res$summaryTable$nbrReads == 2]),
               sort(c("BACH2.0.WT_r.16.C_r.17.C_r.18.C", "BATF2.0.WT_r.61.T_r.62.G", 
                      "BATF2.0.WT_r.19.G_r.20.G_r.21.G", "CEBPB.0.WT_r.34.C_r.35.T_r.36.C",
                      "CEBPD.0.WT_r.37.C_r.39.C", "CEBPD.0.WT_r.67.G_r.69.G", 
                      "CREB3L3.0.WT_r.65.G_r.66.G", "FOSL1.0.WT_r.64.C", 
                      "FOSL2.0.WT_r.38.G_r.39.G", "JUNB.0.WT_r.0.WT", "XBP1.0.WT_r.0.WT")))
  expect_equal(sort(res$summaryTable$mutantNameBaseHGVS[res$summaryTable$nbrReads == 2]),
               sort(c("BACH2:c_r:c.16_18delinsCCC", "BATF2:c_r:c.61_62delinsTG", 
                      "BATF2:c_r:c.19_21delinsGGG", "CEBPB:c_r:c.34_36delinsCTC",
                      "CEBPD:c_r:c.37_39delinsCCC", "CEBPD:c_r:c.67_69delinsGAG", 
                      "CREB3L3:c_r:c.65_66delinsGG", "FOSL1:c_r:c.64G>C", 
                      "FOSL2:c_r:c.38_39delinsGG", "JUNB:c_r:c", "XBP1:c_r:c")))
  expect_equal(sort(res$summaryTable$mutantNameAAHGVS[res$summaryTable$nbrReads == 2]),
               sort(c("BACH2:p_r:p.(Glu6Pro)", "BATF2:p_r:p.(Thr21Trp)", 
                      "BATF2:p_r:p.(Lys7Gly)", "CEBPB:p_r:p.(Lys12Leu)",
                      "CEBPD:p_r:p.(Ala13Pro)", "CEBPD:p_r:p.(Asn23Glu)",
                      "CREB3L3:p_r:p.(Ala22Gly)", "FOSL1:p_r:p.(Ala22Pro)", 
                      "FOSL2:p_r:p.(Ala13Gly)", "JUNB:p_r:p", "XBP1:p_r:p")))
  expect_equal(res$summaryTable$nbrUmis, rep(0L, nrow(res$summaryTable))) ## no UMIs in this experiment
  
  ## ---------------------------------------------------------------------------
  ## Test that we get the same results with useTreeWTmatch = TRUE
  ## ---------------------------------------------------------------------------
  Ldef$useTreeWTmatch <- TRUE
  res2 <- do.call(digestFastqs, Ldef)
  
  expect_equal(res$filterSummary$nbrTotal, res2$filterSummary$nbrTotal)
  expect_equal(res$filterSummary$f1_nbrAdapter, res2$filterSummary$f1_nbrAdapter)
  expect_equal(res$filterSummary$f2_nbrNoPrimer, res2$filterSummary$f2_nbrNoPrimer)
  expect_equal(res$filterSummary$f3_nbrReadWrongLength, res2$filterSummary$f3_nbrReadWrongLength)
  expect_equal(res$filterSummary$f4_nbrNoValidOverlap, res2$filterSummary$f4_nbrNoValidOverlap)
  expect_equal(res$filterSummary$f5_nbrAvgVarQualTooLow, res2$filterSummary$f5_nbrAvgVarQualTooLow)
  expect_equal(res$filterSummary$f6_nbrTooManyNinVar, res2$filterSummary$f6_nbrTooManyNinVar)
  expect_equal(res$filterSummary$f7_nbrTooManyNinUMI, res2$filterSummary$f7_nbrTooManyNinUMI)
  expect_equal(res$filterSummary$f8_nbrTooManyBestWTHits, res2$filterSummary$f8_nbrTooManyBestWTHits)
  expect_equal(res$filterSummary$f9_nbrMutQualTooLow, res2$filterSummary$f9_nbrMutQualTooLow)
  expect_equal(res$filterSummary$f10a_nbrTooManyMutCodons, res2$filterSummary$f10a_nbrTooManyMutCodons)
  expect_equal(res$filterSummary$f10b_nbrTooManyMutBases, res2$filterSummary$f10b_nbrTooManyMutBases)
  expect_equal(res$filterSummary$f11_nbrForbiddenCodons, res2$filterSummary$f11_nbrForbiddenCodons)
  expect_equal(res$filterSummary$f12_nbrTooManyMutConstant, res2$filterSummary$f12_nbrTooManyMutConstant)
  expect_equal(res$filterSummary$f13_nbrTooManyBestConstantHits, res2$filterSummary$f13_nbrTooManyBestConstantHits)
  expect_equal(res$filterSummary$nbrRetained, res2$filterSummary$nbrRetained)
  
  expect_equal(res$summaryTable$nbrReads, res2$summaryTable$nbrReads)
  expect_equal(res$summaryTable$mutantName, res2$summaryTable$mutantName)
  expect_equal(res$summaryTable$sequence, res2$summaryTable$sequence)
  
  ## ---------------------------------------------------------------------------
  ## Test aggregating across groups of wildtype sequences
  ## ---------------------------------------------------------------------------
  Ldef <- list(
      fastqForward = fqt1, fastqReverse = fqt2, 
      mergeForwardReverse = FALSE, 
      minOverlap = 0, maxOverlap = 0, maxFracMismatchOverlap = 0, greedyOverlap = TRUE, 
      revComplForward = FALSE, revComplReverse = FALSE,
      elementsForward = "SPV", elementsReverse = "SPSS",
      elementLengthsForward = c(-1, -1, -1),
      elementLengthsReverse = c(-1, -1, 96, -1),
      adapterForward = "", 
      adapterReverse = "",
      primerForward = "GTCAGGTGGAGGCGGATCC",
      primerReverse = "GAAAAAGGAAGCTGGAGAGA",
      wildTypeForward = leu,
      wildTypeReverse = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTT", 
      constantForward = "", 
      constantReverse = "", 
      avePhredMinForward = 20.0, avePhredMinReverse = 20.0,
      variableNMaxForward = 0, variableNMaxReverse = 0, 
      umiNMax = 0,
      nbrMutatedCodonsMaxForward = 0,
      nbrMutatedCodonsMaxReverse = 0,
      nbrMutatedBasesMaxForward = -1,
      nbrMutatedBasesMaxReverse = -1,
      forbiddenMutatedCodonsForward = "",
      forbiddenMutatedCodonsReverse = "NNW",
      useTreeWTmatch = FALSE,
      mutatedPhredMinForward = 0.0, mutatedPhredMinReverse = 0.0,
      mutNameDelimiter = ".", 
      constantMaxDistForward = -1,
      constantMaxDistReverse = -1,
      umiCollapseMaxDist = 0,
      filteredReadsFastqForward = "",
      filteredReadsFastqReverse = "",
      maxNReads = -1, verbose = FALSE,
      nThreads = 1, chunkSize = 1000, 
      maxReadLength = 1024
  )
  res1 <- do.call(digestFastqs, Ldef)
  leu2 <- leu
  names(leu2)[1:5] <- "ATF2"
  Ldef$wildTypeForward <- leu2
  res2 <- do.call(digestFastqs, Ldef)
  
  expect_identical(res1$filterSummary, res2$filterSummary)
  expect_identical(res1$errorStatistics, res2$errorStatistics)
  expect_equal(nrow(res1$summaryTable), nrow(res2$summaryTable) + 4)
  idx <- which(!res1$summaryTable$mutantName %in% paste0(names(leu)[1:5], ".0.WT_r.0.WT"))
  expect_identical(res1$summaryTable[idx, ], 
                   res2$summaryTable[match(res1$summaryTable$mutantName[idx], 
                                           res2$summaryTable$mutantName), ],
                   ignore_attr = TRUE)
  expect_equal(sum(res1$summaryTable[-idx, "nbrReads"]), 
               res2$summaryTable[res2$summaryTable$mutantName == 
                                     paste0(names(leu2)[1], ".0.WT_r.0.WT"), "nbrReads"])
  
  ## ---------------------------------------------------------------------------
  ## Swap forward and reverse reads, specify max nbr of mutated bases
  ## ---------------------------------------------------------------------------
  Ldef <- list(
      fastqForward = fqt2, fastqReverse = fqt1, 
      mergeForwardReverse = FALSE, 
      minOverlap = 0, maxOverlap = 0, maxFracMismatchOverlap = 0, greedyOverlap = TRUE, 
      revComplForward = FALSE, revComplReverse = FALSE,
      elementsForward = "SPSS", elementsReverse = "SPV", 
      elementLengthsForward = c(-1, -1, 96, -1),
      elementLengthsReverse = c(-1, -1, -1),
      adapterForward = "", 
      adapterReverse = "",
      primerForward = "GAAAAAGGAAGCTGGAGAGA",
      primerReverse = "GTCAGGTGGAGGCGGATCC",
      wildTypeForward = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTT", 
      wildTypeReverse = leu,
      constantForward = "", 
      constantReverse = "", 
      avePhredMinForward = 20.0, avePhredMinReverse = 20.0,
      variableNMaxForward = 0, variableNMaxReverse = 0, 
      umiNMax = 0,
      nbrMutatedCodonsMaxForward = -1,
      nbrMutatedCodonsMaxReverse = -1,
      nbrMutatedBasesMaxForward = 0,
      nbrMutatedBasesMaxReverse = 0,
      forbiddenMutatedCodonsForward = "NNW",
      forbiddenMutatedCodonsReverse = "",
      useTreeWTmatch = FALSE,
      mutatedPhredMinForward = 0.0, mutatedPhredMinReverse = 0.0,
      mutNameDelimiter = ".", 
      constantMaxDistForward = -1,
      constantMaxDistReverse = -1,
      umiCollapseMaxDist = 0,
      filteredReadsFastqForward = "",
      filteredReadsFastqReverse = "",
      maxNReads = -1, verbose = FALSE,
      nThreads = 1, chunkSize = 1000, 
      maxReadLength = 1024
  )
  res1 <- do.call(digestFastqs, Ldef)
  leu2 <- leu
  names(leu2)[1:5] <- "ATF2"
  Ldef$wildTypeReverse <- leu2
  res2 <- do.call(digestFastqs, Ldef)
  
  expect_identical(res1$filterSummary, res2$filterSummary)
  expect_identical(res1$errorStatistics, res2$errorStatistics)
  expect_equal(nrow(res1$summaryTable), nrow(res2$summaryTable) + 4)
  idx <- which(!res1$summaryTable$mutantName %in% paste0("f.0.WT_", names(leu)[1:5], ".0.WT"))
  expect_identical(res1$summaryTable[idx, ], 
                   res2$summaryTable[match(res1$summaryTable$mutantName[idx], 
                                           res2$summaryTable$mutantName), ],
                   ignore_attr = TRUE)
  expect_equal(sum(res1$summaryTable[-idx, "nbrReads"]), 
               res2$summaryTable[res2$summaryTable$mutantName == 
                                     paste0("f.0.WT_", names(leu2)[1], ".0.WT"), "nbrReads"])
  
  ## ---------------------------------------------------------------------------
  ## Too many WT hits, reverse reads
  ## ---------------------------------------------------------------------------
  Ldef <- list(
      fastqForward = fqt1, fastqReverse = fqt2, 
      mergeForwardReverse = FALSE, 
      minOverlap = 0, maxOverlap = 0, maxFracMismatchOverlap = 0, greedyOverlap = TRUE, 
      revComplForward = FALSE, revComplReverse = FALSE,
      elementsForward = "SPV", elementsReverse = "SPSS",
      elementLengthsForward = c(-1, -1, -1),
      elementLengthsReverse = c(-1, -1, 96, -1),
      adapterForward = "", 
      adapterReverse = "",
      primerForward = "GTCAGGTGGAGGCGGATCC",
      primerReverse = "GAAAAAGGAAGCTGGAGAGA",
      wildTypeForward = leu,
      wildTypeReverse = c(A = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTT", B = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCTTA"), 
      constantForward = "", 
      constantReverse = "", 
      avePhredMinForward = 20.0, avePhredMinReverse = 20.0,
      variableNMaxForward = 0, variableNMaxReverse = 0, 
      umiNMax = 0,
      nbrMutatedCodonsMaxForward = 0,
      nbrMutatedCodonsMaxReverse = 0,
      nbrMutatedBasesMaxForward = -1,
      nbrMutatedBasesMaxReverse = -1,
      forbiddenMutatedCodonsForward = "",
      forbiddenMutatedCodonsReverse = "NNW",
      useTreeWTmatch = FALSE,
      mutatedPhredMinForward = 0.0, mutatedPhredMinReverse = 0.0,
      mutNameDelimiter = ".", 
      constantMaxDistForward = -1,
      constantMaxDistReverse = -1,
      umiCollapseMaxDist = 0,
      filteredReadsFastqForward = "",
      filteredReadsFastqReverse = "",
      maxNReads = -1, verbose = FALSE,
      nThreads = 1, chunkSize = 1000, 
      maxReadLength = 1024
  )
  res1 <- do.call(digestFastqs, Ldef)
  
  expect_identical(res1$filterSummary$f8_nbrTooManyBestWTHits, 809L)
  
  ## ---------------------------------------------------------------------------
  ## Too short WT hit, reverse reads
  ## ---------------------------------------------------------------------------
  Ldef <- list(
      fastqForward = fqt1, fastqReverse = fqt2, 
      mergeForwardReverse = FALSE, 
      minOverlap = 0, maxOverlap = 0, maxFracMismatchOverlap = 0, greedyOverlap = TRUE, 
      revComplForward = FALSE, revComplReverse = FALSE,
      elementsForward = "SPV", elementsReverse = "SPVS",
      elementLengthsForward = c(-1, -1, -1),
      elementLengthsReverse = c(-1, -1, 96, -1),
      adapterForward = "", 
      adapterReverse = "",
      primerForward = "GTCAGGTGGAGGCGGATCC",
      primerReverse = "GAAAAAGGAAGCTGGAGAGA",
      wildTypeForward = leu,
      wildTypeReverse = "ATCGCCCGGCTGGAGGAAAAAGTGAAAACCTTGAAAGCTCAGAACTCGGAGCTGGCGTCCACGGCCAACATGCTCAGGGAACAGGTGGCACAGCT", 
      constantForward = "", 
      constantReverse = "", 
      avePhredMinForward = 20.0, avePhredMinReverse = 20.0,
      variableNMaxForward = 0, variableNMaxReverse = 0, 
      umiNMax = 0,
      nbrMutatedCodonsMaxForward = 0,
      nbrMutatedCodonsMaxReverse = 1,
      nbrMutatedBasesMaxForward = -1,
      nbrMutatedBasesMaxReverse = -1,
      forbiddenMutatedCodonsForward = "",
      forbiddenMutatedCodonsReverse = "NNW",
      useTreeWTmatch = FALSE,
      mutatedPhredMinForward = 0.0, mutatedPhredMinReverse = 0.0,
      mutNameDelimiter = ".", 
      constantMaxDistForward = -1,
      constantMaxDistReverse = -1,
      umiCollapseMaxDist = 0,
      filteredReadsFastqForward = "",
      filteredReadsFastqReverse = "",
      maxNReads = -1, verbose = FALSE,
      nThreads = 1, chunkSize = 1000, 
      maxReadLength = 1024
  )
  res1 <- do.call(digestFastqs, Ldef)
  
  expect_identical(res1$filterSummary$f3_nbrReadWrongLength, 439L)
  expect_identical(res1$filterSummary$nbrRetained, 0L)
})
