#' Extracts Tumor Sample Barcodes where the given genes are mutated.
#'
#' @description Extracts Tumor Sample Barcodes where the given genes are mutated.
#'
#' @param maf an \code{\link{MAF}} object generated by \code{\link{read.maf}}
#' @param genes Hogo_Symbol for which sample names to be extracted.
#' @param justNames if TRUE, just returns samples names instead of summarized tables.
#' @param verbose Default TRUE
#' @return list of \code{data.table}s with samples in which given genes are mutated.
#' @examples
#' laml.maf <- system.file("extdata", "tcga_laml.maf.gz", package = "maftools")
#' laml <- read.maf(maf = laml.maf)
#' genesToBarcodes(maf = laml, genes = 'DNMT3A')
#' @export


genesToBarcodes = function(maf, genes = NULL, justNames = FALSE, verbose = TRUE){

  if(is.null(genes)){
    stop('Atleast one gene name is required.')
  }

  dat = createOncoMatrix(m = maf, g = genes)$numericMatrix

  if(length(genes[!genes %in% rownames(dat)]) > 0){
    if(verbose){
      message(paste(genes[!genes %in% rownames(dat)], 'not found in MAF. Excluding it..\n', sep=' '))
    }
    genes = genes[genes %in% rownames(dat)]

    if(length(genes) == 0){
      if(verbose){
        message('No genes left!')
      }
      return(NULL)
    }
  }

  res = list()

  for(i in 1:length(genes)){
    if(ncol(dat) == 1){
      tsbs = colnames(dat)
    }else{
      x = dat[genes[i],]
      tsbs = names(x[x != 0])
    }
    res[i] = list(maf@variant.classification.summary[Tumor_Sample_Barcode %in% tsbs])
    names(res)[i] = genes[i]
  }

  if(justNames){
    res = lapply(X = res, FUN = function(x) as.character(x[,Tumor_Sample_Barcode]))
    return(res)
  }else{
    return(res)
  }
}
