#' Compute glycan traits from a parsed glycan tree
#'
#' Combine residue-level composition traits (see \code{count_residues}),
#' structural traits (see \code{compute_structural_traits}), and
#' user-defined motifs (see \code{compute_userdefined_traits})
#' into a unified trait vector.
#'
#' @param tree A parsed glycan tree from \code{\link{pGlyco3_to_tree}} or \code{\link{wurcs_to_tree}}.
#'
#' @param motifs Optional named list of user-defined glycan motifs.
#'
#' @return A named list of numeric trait values.
#'
#' @examples
#' # Example: parse a pGlyco3-style glycan expression into a tree
#' pGlyco_expr <- "(N(N(H(H(H))(H(H)(H)(H(H))))))"
#'
#' # Convert to glycan tree structure
#' tree <- pGlyco3_to_tree(pGlyco_expr)
#'
#' # Explore parsed nodes and edges
#' tree$node
#' tree$edge
#'
#' # Build igraph representation
#' g <- build_glycan_igraph(tree)
#' plot_glycan_tree(g)
#'
#' # Define user motifs for trait computation
#' user_motifs <- list(
#'   LinearH3 = list(
#'     node = c("H", "H", "H"),
#'     edge = c("a-b", "b-c")
#'   ),
#'   FucBranch = list(
#'     node = c("H", "N", "F"),
#'     edge = c("a-b", "b-c")
#'   )
#' )
#'
#' # Compute glycan structural traits
#' compute_glycan_traits(tree, motifs = user_motifs)
#'
#' @export
compute_glycan_traits <- function(tree, motifs) {
  cnt_trait <- count_residues(tree)
  struct_trait <- compute_structural_traits(tree)
  ud_trait <- compute_userdefined_traits(tree, motifs)

  traits <- c(cnt_trait, struct_trait, ud_trait)
  as.list(traits)
}

#' Append computed glycan traits to a GPSM table
#'
#' Parse each unique glycan structure in a GPSM table, compute its
#' residue and structural traits, and expand the resulting trait
#' columns back to the full GPSM table.
#'
#' @param gpsm A GPSM table from either
#' \code{\link{read_pGlyco3_gpsm}} or \code{\link{read_decipher_gpsm}}.
#'
#' @param from One of `decipher` or `pGlyco3`.
#'
#' @param motifs Optional list of user-defined structural motifs.
#'
#' @return A modified GPSM table with appended trait columns.
#'
#' @keywords internal
#' @noRd
annotate_traits_to_gpsm <- function(gpsm, from, motifs) {
  # Split key GPSM columns and other metadata columns
  glycan <- unique(gpsm$GlycanStructure)

  # Select appropriate glycan-to-tree converter
  if (from == "decipher") {
    get_trait_vector <- function(glycan) {
      glycan_list <- wurcs_to_tree(glycan)
      res_list <- compute_glycan_traits(glycan_list, motifs)
      unlist(res_list)
    }
  } else if (from == "pGlyco3") {
    get_trait_vector <- function(glycan) {
      glycan_list <- pGlyco3_to_tree(glycan)
      res_list <- compute_glycan_traits(glycan_list, motifs)
      unlist(res_list)
    }
  } else {
    stop("`from` must be either 'decipher' or 'pGlyco3'. No traits were computed.")
  }

  # Compute traits for each glycan structure
  traits_list <- pbapply::pblapply(glycan, get_trait_vector)
  traits_columns <- do.call(rbind, traits_list)

  # add trait to psm matrix
  ind <- match(gpsm$GlycanStructure, glycan)
  traits_add <- gpsm$Count * traits_columns[ind, ]
  # combine the both
  gpsm <- gpsm[, !names(gpsm) %in% c("Count", "GlycanStructure"), drop = FALSE]
  gpsm <- cbind(gpsm, traits_add)

  gpsm
}

#' Build a SummarizedExperiment of glycan trait matrices
#'
#' Convert a GPSM table into peptide- or protein-level glycan trait matrices
#' and store them in a \code{SummarizedExperiment} object. Each trait becomes an assay
#' matrix whose rows represent peptides or proteins, and whose columns
#' represent individual GPSMs.
#' This function provides a unified container for downstream analyses such as
#' differential testing and visualization.
#'
#' @param gpsm A GPSM table containing at least:
#'   `Protein`, `Peptide`, `GlycanStructure`, `File`, and `Count`.
#' @param from Character; glycan format used in the GPSM input.
#'   One of `decipher` or `pGlyco3`.
#' @param motifs Optional named list of user-defined motif structures
#'   passed to \link{compute_glycan_traits}.
#' @param level Summarization level. Either `site (peptide)` or `"protein"`.
#' @param meta Data frame of sample metadata with a column `file`
#'   matching `gpsm$File`.
#'
#' @return A `SummarizedExperiment` where each assay is a glycan-trait
#'   matrix (trait × PSM), `rowData` contains peptide/protein names,
#'   and `colData` contains metadata aligned to PSMs.
#'
#'
#' @examples
#' # Load toy GPSM table exported by pGlyco3
#' path <- system.file("extdata", "pGlyco3_gpsm_toyexample.txt",
#'   package = "glycoTraitR"
#' )
#' gpsm_toyexample <- read_pGlyco3_gpsm(path)
#'
#' # Load toy metadata for summarization
#' data("meta_toyexample")
#'
#'
#' # Build glycan trait SummarizedExperiment at protein level
#' trait_se <- build_trait_se(
#'   gpsm = gpsm_toyexample,
#'   from = "pGlyco3",
#'   motifs = NULL,
#'   level = "protein",
#'   meta = meta_toyexample
#' )
#'
#' # Inspect assay names and dimensions
#' SummarizedExperiment::assayNames(trait_se)
#' dim(trait_se)
#'
#' @export
build_trait_se <- function(gpsm, from, motifs = NULL, level, meta) {
  message("adding traits to the gpsm matrix")
  gpsm_mat <- annotate_traits_to_gpsm(gpsm, from, motifs)

  if (level == "site") {
    sel <- "Peptide"
  } else if (level == "protein") {
    sel <- "Protein"
  } else {
    stop("`level` must be either 'site' or 'protein'.")
  }

  gpsm_mat$psm_id <- paste0("psm", seq_len(nrow(gpsm_mat)))

  traits <- setdiff(colnames(gpsm_mat), c("Protein", "Peptide", "File", "psm_id"))
  traits_len <- length(traits)
  get_se_level_trait <- function(i) {
    out <- gpsm_mat[, c(sel, "psm_id", traits[i]), drop = FALSE]
    out <- {
      rr <- unique(out[[sel]])
      cc <- unique(out$psm_id)
      m <- matrix(NA, length(rr), length(cc), dimnames = list(NULL, cc))
      m[cbind(match(out[[sel]], rr), match(out$psm_id, cc))] <- out[[traits[i]]]
      data.frame(stats::setNames(list(rr), sel), m, check.names = FALSE)
    }
    row.names(out) <- out[[1]]
    out[, -1]
  }
  message(sprintf("generating %s trait matrices", level))

  trait_mat_list <- pbapply::pblapply(seq_len(traits_len), get_se_level_trait)
  names(trait_mat_list) <- traits


  # Summarized Experiments
  ind <- match(gpsm_mat$File, meta$file)
  coldata <- meta[ind, ]
  coldata$psm_id <- gpsm_mat$psm_id
  rownames(coldata) <- NULL

  rowdata <- data.frame(
    level = rownames(trait_mat_list[[1]])
  )

  se <- SummarizedExperiment::SummarizedExperiment(
    assays = trait_mat_list,
    rowData = rowdata,
    colData = coldata
  )
  se
}
