% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/input.R
\name{geva.merge.input}
\alias{geva.merge.input}
\alias{geva.read.tables}
\title{GEVA Input Processing and Merge}
\usage{
geva.merge.input(
  ...,
  col.values = "logFC",
  col.pvals = "adj.P.Val",
  col.other = NULL
)

geva.read.tables(
  filenames = NULL,
  dirname = ".",
  col.values = "logFC",
  col.pvals = "adj.P.Val",
  col.other = NULL,
  ...,
  files.pattern = "\\\\.txt$",
  p.value.cutoff = 0.05,
  read.args = list()
)
}
\arguments{
\item{...}{multiple \code{matrix} or \code{data.frame} objects. At least two arguments are required for \code{geva.merge.input}, but it's optional for \code{geva.read.tables}. The optional arguments in \code{geva.read.tables} are also passed to its internal call to \code{geva.merge.input} and \code{\link{geva.input.filter}}.
\cr In addition, the following optional arguments are accepted:
\itemize{
\item{\code{na.val}  : (\code{numeric}) value between \code{0} and \code{1} used as replacement when a p-value column is not present (default is \code{NA})}
\item{\code{use.regex} : (\code{logical}) whether to match the column names using \link[base:regex]{regular expressions} (default is \code{FALSE}) }
\item{\code{verbose} : (\code{logical}) whether to print the current loading and merge progress (default is \code{TRUE})}
}}

\item{col.values}{\code{character} vector, possible name(s) to match the \emph{logFC} column(s) from each table}

\item{col.pvals}{\code{character} vector, possible name(s) to match the p-value column(s) from each table}

\item{col.other}{\code{character} vector, name(s) to match additional columns (\emph{e.g.}, gene symbols). Ignored if \code{NULL}}

\item{filenames}{\code{character} vector with two or more file paths}

\item{dirname}{single \code{character}, base directory containing the input files. Ignored if \code{filenames} is specified}

\item{files.pattern}{single \code{character}, pattern used to filter the files inside \code{dirname}. Ignored if \code{filenames} is specified}

\item{p.value.cutoff}{\code{numeric} (\code{0} to \code{1}), initial p-value threshold. Rows entirely composed by p-values above this cutoff (\emph{i.e.}, no significant \emph{logFC}) are removed after the final merge. Ignored if \code{NA} or \code{NULL}}

\item{read.args}{\code{list} of additional arguments passed to \link[utils:read.table]{utils::read.table}}
}
\value{
A \code{\linkS4class{GEVAInput}} object
}
\description{
Functions to read, load, and concatenate the experimental comparisons from the data input. This is the initial step to proceed with any GEVA analysis.
}
\details{
The \code{geva.merge.input} function takes multiple tables as arguments (\emph{e.g.}, \code{matrix} or \code{data.frame} objects), extracts the \emph{logFC} columns from each table and merges them into a single \code{\linkS4class{GEVAInput}} dataset.

The column names are specified in the \code{col.values} and \code{col.pvals} arguments (\code{character}) and must correctly match the column names for \emph{logFC} and p-value columns, respectively, in the inputs to be extracted.
Multiple values for column names can also be specified as valid name possibilities if they differ among the tables.

The function \code{geva.merge.input} reads multiple tab-delimited text files containing, extracts the \emph{logFC} columns from each table and merges into a single \code{\linkS4class{GEVAInput}} dataset.
}
\note{
The inclusion of p-value columns is not technically required, but strongly recommended as they improve the statistical accuracy in the summarization steps. If the p-value (or adjusted p-value) columns are present, their values are converted to weights by applying \code{1 - pvalue} for each \code{pvalue} element, otherwise an optional \code{na.val} optional argument can specified as replacement to the absent values (default is \code{NA}). Weights are used to accomodate the central \emph{logFC} values towards the most significant observations and penalize potential statistical innacuracies.
}
\examples{
### EXAMPLE 1
## geva.merge.input example with three randomly generated tables
## (For demonstration purposes only)

# Number of rows
n <- 10000

# Random row (probe) names
probnms <- sprintf("PROBE_\%s", 1:n)

# Random gene names (optional)
genenms <- paste0(sprintf("GENE_\%s", 1:n), LETTERS[1:n \%\% (length(LETTERS)+1)])

# Random table 1
dt1 <- data.frame(row.names=probnms,
                  logfc=(rnorm(n, 0, sd=2) * rnorm(n, 0, sd=0.5)),
                  pvalues = runif(n, max=0.08),
                  genesymbol = genenms)
# Random table 2
dt2 <- data.frame(row.names=probnms,
                  logfc=(rnorm(n, 0, sd=2) * rnorm(n, 0, sd=0.5)),
                  pvalues = runif(n, max=0.08),
                  genesymbol = genenms)
# Random table 3
dt3 <- data.frame(row.names=probnms,
                  logfc=(rnorm(n, 0, sd=2) * rnorm(n, 0, sd=0.5)),
                  pvalues = runif(n, max=0.08),
                  genesymbol = genenms)

# Merges the three tables
ginput <- geva.merge.input(exp1=dt1, exp2=dt2, exp3=dt3,
                           col.values="logfc",
                           col.pvals="pvalues",
                           col.other="genesymbol")

# Prints the first rows from the merged table
print(head(ginput))               # values
print(head(inputweights(ginput))) # weights

# ---
\dontrun{

### EXAMPLE 2
## geva.read.tables example with three tab-delimited files

# Table file examples. Each one has 3 columns: "logfc", "pvalues", and "genesymbol"
# Replace it with your tab-delimited files (e.g. exported from limma's topTable)
fnames <- c("dt1.txt", "dt2.txt", "dt3.txt")

ginput <- geva.read.tables(fnames,
                           col.values="logfc",
                           col.pvals="pvalues",
                           col.other="genesymbol")

# Prints the first rows from the merged table
print(head(ginput))               # values
print(head(inputweights(ginput))) # weights


# ---

### EXAMPLE 3
## geva.read.tables example with tab-delimited files in a directory

# Directory name (replace it with a directory containing the table files)
dirnm <- "C:/User/robertplant123/Documents/R/gevaexamples"

# In this example, table files contain 3 columns: "logfc", "pvalues", and "genesymbol"
# Reads all txt files in the directory
ginput <- geva.read.tables(dirname=dirnm,
                           col.values="logfc",
                           col.pvals="pvalues",
                           col.other="genesymbol")

# (Optional step)
# Let's assume that all table file names start with "dt" and ends with the ".txt" extension,
# such as dt1.txt, dt2.txt and so on...
fname_pattern <- c("^dt.+?\\\\.txt$")  # Defines a RegEx pattern to find the files
# Loads only files that match the file name pattern
ginput <- geva.read.tables(dirname=dirnm,
                           files.pattern=fname_pattern,
                           col.values="logfc",
                           col.pvals="pvalues",
                           col.other="genesymbol")

# Prints the first rows from the merged table
print(head(ginput))               # values
print(head(inputweights(ginput))) # weights
}

}
