% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculations_Qentropy.R
\name{Qentropy}
\alias{Qentropy}
\title{Compute Q-Entropy using existing row-normalized data + Entropy}
\usage{
Qentropy(x, assay_name = "Entropy", new_assay_name = "Qentropy")
}
\arguments{
\item{x}{A data.frame (already processed by 'entropy()')
or a SummarizedExperiment (already processed by 'entropy()').}

\item{assay_name}{(SummarizedExperiment only) The name of the assay
whose row-normalized data will be replaced by Q-values. If NULL,
uses the first assay.}

\item{new_assay_name}{If you prefer to store Q-values in a
*new* assay, provide a name. By default 'Qentropy'}
}
\value{
\itemize{
  \item If \code{x} is a data.frame: returns the same data.frame with numeric
   columns replaced by \eqn{Q_{ij}} values and \code{Entropy} column removed.
  \item If \code{x} is a SummarizedExperiment: returns the same object with
    the specified assay replaced by \eqn{Q_{ij}} values (or a new assay
    if \code{new_assay_name} is set) and \code{rowData(x)$Entropy} removed.
    }
}
\description{
#' Transform entropy scores into categorical entropy scores
\eqn{Q_{ij} = \mathrm{Entropy}_i - \log_2(x_{ij})}, or \code{Inf} if
\eqn{x_{ij} == 0}.

 @details
For each row \eqn{i} and column \eqn{j}, \eqn{Q_{ij}} is defined as
\eqn{\mathrm{Entropy}_i - \log_2\bigl(x_{ij}\bigr)} if \eqn{x_{ij}} is
positive, or \code{Inf} otherwise.
}
\examples{

library(SummarizedExperiment)
library(airway)
data('airway')

se = airway

# Only use a random subset of 1000 rows
set.seed(123)
idx <- sample(seq_len(nrow(se)), size = min(1000, nrow(se)))
se <- se[idx, ]

# -------------------------------
# 1) Using a data.frame
# -------------------------------
df = assay(se) |> as.data.frame()

## Entropy needs to be calculated first
df = entropy(df)

## Then you can apply the Qentropy function
df = Qentropy(df)

head(df)

# -------------------------------
# 2) Using a SummarizedExperiment
# -------------------------------

## Calculate Entropy first
se2 = entropy(se, new_assay_name = 'Entropy')

## Transform entropy into Qentropy. new_assay_name specify a new assay
## where data is going to be stored. Assay_name must have Entropy transformed
values
## By default, the function will look for an assay_name 'Entropy' and assign
## a new assay to 'Qentropy'
se2 = Qentropy(se2, new_assay_name = 'Qentropy', assay_name = 'Entropy')

se2

}
