\name{plotDI}
\alias{plotDI}
\alias{rotDI}

\title{Construct a plaid plot of differential interactions}

\description{Plot differential interactions in a plaid format with informative coloring.}

\usage{
plotDI(data, fc, first.region, second.region=first.region, col.up="red", 
    col.down="blue", background="grey70", zlim=NULL, xlab=NULL, ylab=NULL, 
    diag=TRUE, ...)

rotDI(data, fc, region, col.up="red", col.down="blue", background="grey70", 
    zlim=NULL, xlab=NULL, max.height=NULL, ylab="Gap", ...)
}

\arguments{
	\item{data}{an InteractionSet object}
	\item{fc}{a numeric vector of log-fold changes}
	\item{first.region}{a \code{GRanges} object of length 1 specifying the first region}
	\item{second.region}{a \code{GRanges} object of length 1 specifying the second region}
	\item{region}{a \code{GRanges} object of length 1 specifying the region of interest}
	\item{col.up}{any type of R color to describe the maximum color for positive log-fold changes}
	\item{col.down}{any type of R color to describe the maximum color for negative log-fold changes}
	\item{background}{any type of R color, specifying the background color of the interaction space}
	\item{zlim}{a numeric scalar indicating the maximum absolute log-fold change}
	\item{xlab}{character string for the x-axis label on the plot, defaults to the first chromosome name}
    \item{max.height}{a numeric scalar indicating the y-axis limit for \code{rotPlaid}}
	\item{ylab}{character string for the y-axis label on the plot, defaults to the second chromosome name in \code{plotDI}}
	\item{diag}{a logical scalar specifying whether boxes should be shown above the diagonal for intra-chromosomal plots in \code{plotDI}}
	\item{...}{other named arguments to be passed to \code{\link{plot}}}
}	

\details{
The \code{plotDI} function constructs a plaid plot on the current graphics device.
The intervals of \code{first.region} and \code{second.region} are represented by the x- and y-axes, respectively.
Each bin pair is represented by a box in the plotting space, where each side of the box represents a bin.
Plotting space that is not covered by any bin pair is shown in \code{background}.

The color of the box depends on the magnitude and sign of the log-fold change in \code{fc}.
Positive log-FCs will range from white to \code{col.up}, whereas negative log-FCs will range from white to \code{col.down}.
The chosen color is proportional to the magnitude of the log-FC, and the most extreme colors are only obtained at the maximum absolute log-FC in \code{fc}.
The maximum value can be capped at \code{zlim} for better resolution of small log-FCs.

If \code{diag=TRUE}, boxes will also be plotted above the diagonal for intra-chromosomal plots. 
This is set as the default to avoid confusion when \code{first.region} is not set as the anchor range, i.e., it has a lower sorting order than \code{second.region}.
However, this can also be turned off to reduce redundancy in visualization around the diagonal.

The \code{rotDI} function constructs a rotated plot of differential interactions, for visualization of local changes.
See \code{\link{rotPlaid}} for more details. 
}

\value{
A (rotated) plaid plot of differential interactions is produced on the current graphics device. 
A function is also invisibly returned that converts log-FCs into colors.
This is useful for coordinating the colors, e.g., when constructing a separate color bar.
}

\author{Aaron Lun}

\examples{
# Setting up the objects.
a <- 10
b <- 20
regions <- GRanges(rep(c("chrA", "chrB"), c(a, b)), IRanges(c(1:a, 1:b), c(1:a, 1:b)),
    seqinfo=Seqinfo(seqlengths=c(chrA=a, chrB=b), seqnames=c("chrA", "chrB")))

set.seed(3423)
all.anchor1 <- sample(length(regions), 500, replace=TRUE)
all.anchor2 <- as.integer(runif(500, 1, all.anchor1+1))
out <- InteractionSet(matrix(0, 500, 1), colData=DataFrame(lib.size=1000),
        GInteractions(anchor1=all.anchor1, anchor2=all.anchor2, 
            regions=regions, mode="reverse"), metadata=List(width=1))
fc <- runif(nrow(out), -2, 2)

# Constructing intra-chromosomal DI plots around various regions
plotDI(out, fc, first.region=GRanges("chrA", IRanges(1, 10)), 
    second.region=GRanges("chrA", IRanges(1, 10)), diag=TRUE)
plotDI(out, fc, first.region=GRanges("chrA", IRanges(1, 10)), 
    second.region=GRanges("chrA", IRanges(1, 10)), diag=FALSE)

# Constructing inter-chromosomal DI plots around various regions
xxx <- plotDI(out, fc, first.region=GRanges("chrB", IRanges(1, 10)), 
    second.region=GRanges("chrA", IRanges(1, 20)), diag=TRUE)
plotDI(out, fc, first.region=GRanges("chrB", IRanges(1, 100)), 
    second.region=GRanges("chrA", IRanges(1, 200)), diag=TRUE, zlim=5)

# Making colorbars.
xxx((-10):10/10)
xxx((-20):20/20)

# Rotated.
rotDI(out, fc, region=GRanges("chrA", IRanges(1, 200)))
rotDI(out, fc, region=GRanges("chrB", IRanges(1, 200)))
}

\seealso{
	\code{\link{plotPlaid}},
	\code{\link{rotPlaid}},
	\code{\link{squareCounts}}
}

\references{
Lieberman-Aiden E et al. (2009). Comprehensive Mapping of Long-Range Interactions Reveals Folding Principles of the Human Genome. \emph{Science} 326, 289-293. 
}

\keyword{visualization}
