#' @title Filter genes by FDR within a specific condition
#' @description
#' Filters a list of genes to retain only those that meet a specified
#' False Discovery Rate (FDR) threshold. The filtering is applied to one or more
#' replicates within a single experimental condition, based on the `_FDR` columns
#' which must be present in the data (e.g., as generated by
#' `load_data_genes(calculate_fdr = TRUE)`).
#'
#' @param data A `DamIDResults` object or the `list` returned by `load_data_genes()`.
#'   The data object must contain an `occupancy` table with FDR columns (e.g., `SampleA_FDR`).
#' @param fdr A numeric value between 0 and 1 specifying the FDR cutoff. Loci with an
#'   FDR less than or equal to this value will be considered. (Default: 0.05)
#' @param condition A character string that identifies the experimental condition to filter on.
#'   This string should uniquely match the relevant sample columns (e.g., "L4" will match
#'   "L4_rep1_FDR" and "L4_rep2_FDR"). If `data` is a `DamIDResults` object, this can
#'   be either the internal identifier or the display name for the condition.
#' @param which A character string, either `"any"` (the default) or `"all"`.
#'   \itemize{
#'     \item If `"any"`, a gene is kept if it meets the `fdr` threshold in at least one
#'       replicate of the specified `condition`.
#'     \item If `"all"`, a gene is kept only if it meets the `fdr` threshold in all
#'       replicates of the specified `condition`.
#'   }
#'
#' @return A `data.frame` containing the `gene_name` and `gene_id` of the genes
#'   that passed the filter. If no genes pass, an empty `data.frame` is returned.
#'
#' @details
#' This function is primarily used in workflows involving RNA Polymerase TaDa data, where
#' an FDR is calculated for gene occupancy to determine if a gene is actively transcribed.
#' It allows users to identify genes in a single condition that can be considered to be
#' expressed (i.e. RNA Pol occupancy is significantly greater than background).
#'
#' Note that while this is an effective proxy for gene expression, there are edge cases
#' (e.g. paused polymerase, short genes directly adjacent to an expressed gene TSS or TES)
#' where a gene may have significant occupancy but not, in fact, be transcribed.
#'
#' The function locates the relevant FDR columns in the `occupancy` table by searching for
#' column names that end with `_FDR` and also contain the `condition` string.
#'
#' @export
#' @examples
#' # Create a mock data object with an occupancy table containing FDR values,
#' # similar to the output of `load_data_genes(calculate_fdr = TRUE)`.
#' .create_mock_fdr_data <- function() {
#'     occupancy_df <- data.frame(
#'         gene_name = c("geneA", "geneB", "geneC", "geneD"),
#'         gene_id = c("FBgn01", "FBgn02", "FBgn03", "FBgn04"),
#'         L4_rep1_FDR = c(0.01, 0.10, 0.04, 0.06),
#'         L4_rep2_FDR = c(0.03, 0.02, 0.50, 0.07),
#'         L5_rep1_FDR = c(0.80, 0.90, 0.01, 0.02),
#'         row.names = c("geneA", "geneB", "geneC", "geneD")
#'     )
#'     list(occupancy = occupancy_df, test_category = "expressed")
#' }
#' mock_data <- .create_mock_fdr_data()
#'
#' # Example 1: Get genes with FDR <= 0.05 in ANY L4 replicate.
#' # geneA (0.01, 0.03), geneB (0.02), and geneC (0.04) pass.
#' expressed_in_L4_any <- filter_genes_by_fdr(
#'     mock_data,
#'     fdr = 0.05,
#'     condition = "L4",
#'     which = "any"
#' )
#' print(expressed_in_L4_any)
#'
#' # Example 2: Get genes with FDR <= 0.05 in ALL L4 replicates.
#' # Only geneA (0.01, 0.03) passes.
#' expressed_in_L4_all <- filter_genes_by_fdr(
#'     mock_data,
#'     fdr = 0.05,
#'     condition = "L4",
#'     which = "all"
#' )
#' print(expressed_in_L4_all)
#'
#' # Example 3: Get genes with FDR <= 0.05 in any L5 replicate.
#' # geneC (0.01) and geneD (0.02) pass.
#' expressed_in_L5 <- filter_genes_by_fdr(
#'     mock_data,
#'     fdr = 0.05,
#'     condition = "L5",
#'     which = "any"
#' )
#' print(expressed_in_L5)
filter_genes_by_fdr <- function(data, fdr = 0.05, condition, which = "any") {
    # Input validation
    if (!is(data, "DamIDResults") && !(is.list(data) && "occupancy" %in% names(data))) {
        stop("'data' must be a DamIDResults object or a list from load_data_genes().")
    }
    if (!is.numeric(fdr) || fdr < 0 || fdr > 1) {
        stop("'fdr' must be a numeric value between 0 and 1.")
    }
    if (!is.character(condition) || length(condition) != 1) {
        stop("'condition' must be a single character string.")
    }
    which <- match.arg(which, c("any", "all"))

    # Extract occupancy data and condition map
    occupancy_df <- if (is(data, "DamIDResults")) inputData(data)$occupancy else data$occupancy
    cond_map <- if (is(data, "DamIDResults")) conditionNames(data) else NULL

    # Centralised validation checks
    if (is.null(occupancy_df)) {
        stop("Could not find 'occupancy' data in the input object.")
    }
    if (!all(c("gene_name", "gene_id") %in% colnames(occupancy_df))) {
        stop("The 'occupancy' table must contain 'gene_name' and 'gene_id' columns.")
    }
    if (is.null(rownames(occupancy_df))) {
        stop("The 'occupancy' table is missing row names, cannot perform filtering.")
    }

    common_cols <- c("gene_id","gene_name","nfrags","name")
    sample_and_fdr_cols <- setdiff(colnames(occupancy_df),common_cols)

    # Find relevant FDR columns
    all_fdr_cols <- grep("_FDR$", sample_and_fdr_cols, value = TRUE)
    if (length(all_fdr_cols) == 0) {
        warning("No '_FDR' columns found in the data. Returning an empty data frame.")
        return(data.frame(gene_name = character(0), gene_id = character(0)))
    }
    all_sample_cols <- setdiff(sample_and_fdr_cols,all_fdr_cols)

    # Determine the internal condition identifier to search for
    internal_condition_id <- condition
    if (!is.null(cond_map) && (condition %in% names(cond_map))) {
        internal_condition_id <- cond_map[condition]
    }

    # Filter FDR columns to those belonging to the specified condition
    relevant_fdr_cols <- grep(internal_condition_id, all_fdr_cols, value = TRUE, fixed = TRUE)
    relevant_sample_cols <- grep(internal_condition_id, all_sample_cols, value = TRUE, fixed = TRUE)

    if (length(relevant_fdr_cols) == 0) {
        warning(sprintf("No FDR columns found matching the condition '%s'.", condition))
        return(data.frame(gene_name = character(0), gene_id = character(0)))
    }
    message(
        sprintf("Found %d FDR columns for condition '%s': %s",
                length(relevant_fdr_cols), condition, paste(relevant_fdr_cols, collapse = ", "))
    )

    # Perform filtering
    fdr_subset <- occupancy_df[, relevant_fdr_cols, drop = FALSE]

    filter_fun <- if (which == "any") {
        function(row) any(row <= fdr, na.rm = TRUE)
    } else {
        function(row) !any(is.na(row)) && all(row <= fdr, na.rm = TRUE)
    }

    keep_indices <- apply(fdr_subset, 1, filter_fun)

    if (sum(keep_indices, na.rm = TRUE) == 0) {
        message(sprintf("No genes passed the FDR <= %s filter for '%s' with rule '%s'.", fdr, condition, which))
        return(data.frame(gene_name = character(0), gene_id = character(0)))
    }

    # Format and return
    result_df <- occupancy_df[keep_indices, c("gene_name", "gene_id", relevant_sample_cols,relevant_fdr_cols), drop = FALSE]
    result_df$avg_occ <- apply(result_df[relevant_sample_cols],1,mean)
    result_df$min_fdr <- apply(result_df[relevant_fdr_cols],1,min)
    result_df <- result_df[c("gene_name", "gene_id","avg_occ","min_fdr")]

    message(
        sprintf("%d genes/loci passed the FDR <= %s filter for condition '%s' with rule '%s'.",
                nrow(result_df), fdr, condition, which)
    )

    return(result_df)
}


#' @title Filter Loci Based on Gene Expression FDR
#' @description Internal helper to filter a list of locus identifiers based on an FDR
#'   threshold applied to `_FDR` columns from the two conditions being compared.
#' @param diff_results A `DamIDResults` object from which to get occupancy data
#'   and condition information.
#' @param fdr_filter_threshold Numeric threshold for FDR filtering.
#' @param row_names_to_filter (Optional) A character vector of locus identifiers (row names)
#'   to be filtered. (Default: NULL, will use all rownames from table)
#' @return A character vector containing the subset of `row_names_to_filter`
#'   that pass the FDR threshold. If filtering is not possible (e.g., no FDR
#'   columns found), it returns the original `row_names_to_filter` vector
#'   along with a warning.
#' @noRd
filter_on_fdr <- function(diff_results, fdr_filter_threshold, row_names_to_filter=NULL) {
    input_data <- inputData(diff_results)
    occupancy_df <- input_data$occupancy

    if (is.null(occupancy_df)) {
        warning("fdr_filter_threshold was set, but no 'occupancy' data found. Returning all loci unfiltered.")
        return(row_names_to_filter)
    }

    all_fdr_cols <- grep("_FDR$", colnames(occupancy_df), value = TRUE)

    if (length(all_fdr_cols) == 0) {
        warning("fdr_filter_threshold was set, but no '_FDR' columns were found. Returning all loci unfiltered.")
        return(row_names_to_filter)
    }

    if (is.null(row_names_to_filter)) {
        row_names_to_filter <- rownames(analysisTable(diff_results))
    }

    relevant_fdr_cols <- character(0)
    if (!is.null(input_data$matched_samples) && is.list(input_data$matched_samples)) {
        message("Using pre-matched samples for FDR filtering.")
        matched_samples <- unlist(input_data$matched_samples)
        fdr_col_names <- paste0(matched_samples, "_FDR")
        relevant_fdr_cols <- intersect(fdr_col_names, all_fdr_cols)
    } else {
        # Fallback to old regex-based method
        warning("Falling back to regex-based FDR column matching. This may be unreliable with complex condition patterns.")
        cond_identifiers <- as.character(conditionNames(diff_results))
        pattern <- paste(cond_identifiers, collapse = "|")
        relevant_fdr_cols <- grep(pattern, all_fdr_cols, value = TRUE)
    }

    if (length(relevant_fdr_cols) == 0) {
        cond_display <- names(conditionNames(diff_results))
        warning(sprintf(
            "fdr_filter_threshold was set, but no '_FDR' columns matching conditions '%s' or '%s' were found. Returning all loci unfiltered.",
            cond_display[1], cond_display[2]
        ))
        return(row_names_to_filter)
    }

    # Align occupancy data using the provided row names
    aligned_occ_df <- occupancy_df[row_names_to_filter, , drop = FALSE]

    if (anyNA(rownames(aligned_occ_df))) {
        stop("A mismatch in row names between the data to be filtered and occupancy data was detected. Cannot perform FDR filtering.")
    }

    fdr_subset <- aligned_occ_df[, relevant_fdr_cols, drop = FALSE]

    # For each locus, check if ANY relevant FDR value is <= threshold
    keep_indices <- apply(fdr_subset, 1, function(row) {
        any(row <= fdr_filter_threshold, na.rm = TRUE)
    })

    filtered_row_names <- row_names_to_filter[keep_indices]

    message(sprintf(
        "Filtering universe: %d of %d total loci pass gene expression FDR <= %.3f.",
        length(filtered_row_names), length(row_names_to_filter), fdr_filter_threshold
    ))

    return(filtered_row_names)
}

