% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reducedMNN.R
\name{reducedMNN}
\alias{reducedMNN}
\title{MNN correction in reduced dimensions}
\usage{
reducedMNN(
  ...,
  batch = NULL,
  k = 20,
  prop.k = NULL,
  restrict = NULL,
  ndist = 3,
  merge.order = NULL,
  auto.merge = FALSE,
  min.batch.skip = 0,
  BNPARAM = KmknnParam(),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{...}{One or more matrices of low-dimensional representations where rows are cells and columns are dimensions.
Each object should contain the same number of columns, corresponding to the same dimensions.
These should have been generated by a single call to \code{\link{multiBatchPCA}}.

If multiple objects are supplied, each object is assumed to contain all and only cells from a single batch.
If a single object is supplied, \code{batch} should also be specified.

Alternatively, any number of lists of such objects.
this is flattened as if the objects inside each list were passed directly to \code{...}.}

\item{batch}{A vector or factor specifying the batch of origin for all cells when only a single object is supplied in \code{...}.
This is ignored if multiple objects are present.}

\item{k}{An integer scalar specifying the number of nearest neighbors to consider when identifying MNNs.}

\item{prop.k}{A numeric scalar in (0, 1) specifying the proportion of cells in each dataset to use for mutual nearest neighbor searching.
If set, the number of nearest neighbors used for the MNN search in each batch is redefined as \code{max(k, prop.k*N)} where \code{N} is the number of cells in that batch.}

\item{restrict}{A list of length equal to the number of objects in \code{...}.
Each entry of the list corresponds to one batch and specifies the cells to use when computing the correction.}

\item{ndist}{A numeric scalar specifying the threshold beyond which neighbours are to be ignored when computing correction vectors.
Each threshold is defined as a multiple of the number of median distances.}

\item{merge.order}{An integer vector containing the linear merge order of batches in \code{...}.
Alternatively, a list of lists representing a tree structure specifying a hierarchical merge order.}

\item{auto.merge}{Logical scalar indicating whether to automatically identify the \dQuote{best} merge order.}

\item{min.batch.skip}{Numeric scalar specifying the minimum relative magnitude of the batch effect, 
below which no correction will be performed at a given merge step.}

\item{BNPARAM}{A \linkS4class{BiocNeighborParam} object specifying the nearest neighbor algorithm.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying whether the PCA and nearest-neighbor searches should be parallelized.}
}
\value{
A \linkS4class{DataFrame} is returned where each row corresponds to a cell, containing:
\itemize{
\item \code{corrected}, the matrix of corrected low-dimensional coordinates for each cell.
\item \code{batch}, the Rle specifying the batch of origin for each row.
}
Cells in the output object are always ordered in the same manner as supplied in \code{...}.
The metadata on this object is the same as that in the output of \code{\link{fastMNN}}.
}
\description{
MNN correction in reduced dimensions
}
\details{
\code{reducedMNN} performs the same operations as \code{\link{fastMNN}} but assumes that the PCA has already been performed.
This is useful as the PCA (via \code{\link{multiBatchPCA}}) is often the most time-consuming step.
By performing the PCA once, \code{reducedMNN} allows the MNN correction to be quickly repeated with different parameters.

\code{reducedMNN} operates on the same principles as \code{\link{fastMNN}}, 
so users are referred to the documentation for the latter for more details on the effect of each of the arguments.
Obviously, any arguments pertaining to gene-based steps in \code{\link{fastMNN}} are not relevant here.

Note that \code{\link{multiBatchPCA}} will not perform cosine-normalization, 
so it is the responsibility of the user to cosine-normalize each batch beforehand with \code{\link{cosineNorm}} to recapitulate results of \code{\link{fastMNN}} with \code{cos.norm=TRUE}.
In addition, \code{\link{multiBatchPCA}} must be run on all samples at once, to ensure that all cells are projected to the same low-dimensional space.
}
\examples{
B1 <- matrix(rnorm(10000), nrow=50) # Batch 1 
B2 <- matrix(rnorm(10000), nrow=50) # Batch 2

# Corrected values equivalent to fastMNN().
cB1 <- cosineNorm(B1)
cB2 <- cosineNorm(B2)
pcs <- multiBatchPCA(cB1, cB2)
mnn.out <- reducedMNN(pcs[[1]], pcs[[2]])

mnn.out

}
\seealso{
\code{\link{multiBatchPCA}}, to obtain the values to be corrected.

\code{\link{fastMNN}}, for the version that operates on gene-expression values.

\code{\link{clusterMNN}}, for an application on cluster centroids.
}
\author{
Aaron Lun
}
