% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quickCorrect.R
\name{quickCorrect}
\alias{quickCorrect}
\title{Quickly perform batch correction}
\usage{
quickCorrect(
  ...,
  batch = NULL,
  restrict = NULL,
  correct.all = TRUE,
  assay.type = "counts",
  PARAM = FastMnnParam(),
  multi.norm.args = list(),
  precomputed = NULL,
  model.var.args = list(),
  hvg.args = list(n = 5000)
)
}
\arguments{
\item{...}{One or more matrix-like objects containing single-cell gene expression matrices.
Alternatively, one or more \linkS4class{SingleCellExperiment} objects can be supplied.

If multiple objects are supplied, each object is assumed to contain all and only cells from a single batch.
Objects of different types can be mixed together.
If a single object is supplied, \code{batch} should also be specified.}

\item{batch}{A factor specifying the batch of origin for each cell if only one batch is supplied in \code{...}.
This will be ignored if two or more batches are supplied.}

\item{restrict}{A list of length equal to the number of objects in \code{...}.
Each entry of the list corresponds to one batch and specifies the cells to use when computing the correction.}

\item{correct.all}{A logical scalar indicating whether to return corrected expression values for all genes, even if \code{subset.row} is set.
Used to ensure that the output is of the same dimensionality as the input.}

\item{assay.type}{A string or integer scalar specifying the assay to use for correction.
Only used for SingleCellExperiment inputs.}

\item{PARAM}{A \linkS4class{BatchelorParam} object specifying the batch correction method to dispatch to, and the parameters with which it should be run.
\linkS4class{ClassicMnnParam} will dispatch to \code{\link{mnnCorrect}};
\linkS4class{FastMnnParam} will dispatch to \code{\link{fastMNN}};
\linkS4class{RescaleParam} will dispatch to \code{\link{rescaleBatches}};
and \linkS4class{RegressParam} will dispatch to \code{\link{regressBatches}}.}

\item{multi.norm.args}{Named list of further arguments to pass to \code{\link{multiBatchNorm}}.}

\item{precomputed}{List of \link{DataFrame}s containing precomputed variance modelling results.
This should be a list of the same length as the number of entries in \code{...},
and each should have the same row names as the corresponding entry of \code{...}.}

\item{model.var.args}{Named list of further arguments to pass to \code{\link[scran]{modelGeneVar}}.}

\item{hvg.args}{Named list of further arguments to pass to \code{\link[scran]{getTopHVGs}}.
By default, we take the top 5000 genes with the highest variances.}
}
\value{
A list containing:
\itemize{
\item \code{dec}, a \linkS4class{DataFrame} containing the combined variance modelling results across all batches.
\item \code{hvgs}, a character vector of the genes selected to use in the correction.
\item \code{corrected}, a \linkS4class{SingleCellExperiment} containing the corrected values across all cells in all batches.
}
}
\description{
Quickly perform batch correction by intersecting the gene features, normalizing and log-transforming,
modelling per-gene variances and identifying highly variable genes, and then applying the correction algorithm of choice.
}
\details{
This function wraps the sequence of typical steps required to obtain corrected expression values.'
\enumerate{
\item Intersecting each batch to the universe of common features with \code{\link{intersectRows}}.
\item Applying normalization and log-transformation to the batches with \code{\link{multiBatchNorm}}.
\item Modelling the per-gene variance with \code{\link[scran]{modelGeneVar}}.
If \code{precomputed} is supplied, the precomputed results are used instead.
\item Identifying highly variable genes with \code{\link[scran]{getTopHVGs}}.
These genes will be used in the correction, though corrected values for all genes can be returned by setting \code{correct.all=TRUE}.
\item Applying the batch correction algorithm of choice with \code{\link{batchCorrect}}, as specified by \code{PARAM}.
}

The default of \code{correct.all=TRUE} differs from that of other functions.
This is because the subsetting to HVGs is done internally here, and we avoid surprises by returning results for all genes in the input object(s).
In contrast, the other functions require explicit subsetting via \code{subset.row=} and it is expected that users will set \code{correct.all=} if all genes are desired.
}
\examples{
d1 <- matrix(rnbinom(50000, mu=10, size=1), ncol=100)
sce1 <- SingleCellExperiment(list(counts=d1))
sizeFactors(sce1) <- runif(ncol(d1))
rownames(sce1) <- paste0("GENE", 1:500)

d2 <- matrix(rnbinom(20000, mu=50, size=1), ncol=40)
sce2 <- SingleCellExperiment(list(counts=d2))
sizeFactors(sce2) <- runif(ncol(d2))
rownames(sce2) <- paste0("GENE", 201:700)

# Fire and forget:
set.seed(1000)
output <- quickCorrect(sce1, sce2) 
output$corrected

}
\seealso{
\code{\link{intersectRows}}, for the intersection to obtain a universe of genes.

\code{\link{multiBatchNorm}}, to perform the normalization.

\code{\link[scran]{modelGeneVar}} and \code{\link[scran]{getTopHVGs}}, to identify the top HVGs for use in correction.

\code{\link{batchCorrect}}, to dispatch to the desired correction algorithm.
}
\author{
Aaron Lun
}
