% NOT generated by roxygen2
% please edit documentation here
\name{ri_plot_peak}
\alias{ri_plot_peak}
\title{Plot peak RI across samples - low level interface}
\usage{
ri_plot_peak(obj, massValues, timeRange, useRT = TRUE, showRT = useRT, sizefun = NULL, plot = TRUE, ...)
}
\description{
    Plot retention index or time of peaks from RI files or samples. This is a low level
    interface to \code{\link{plotPeakRI}} that not relies on a metabolite library.
    This function is useful for quick peak data visualization.
}
\arguments{
    \item{obj}{Either a \code{\linkS4class{tsSample}} object or a character vector of paths to RI files.}
    \item{massValues}{A numeric vector representing \emph{m/z} values.}
    \item{timeRange}{A numeric vector of length 2 representing the lower and upper time limits.}
    \item{useRT}{Logical. If \code{TRUE} (default), the time range is expressed in seconds,
        otherwise if \code{FALSE}, the time range is in retention time index units.}
    \item{showRT}{Logical. Whether the \emph{x}-axis of the resulting plot should represent seconds
        (\code{TRUE}) or retention index units (\code{FALSE}). This allows, for example, to search
        for a time range in seconds and display a graph in index units.}
    \item{sizefun}{A function that could be used to scale the point size of the plot via the
        parameter `cex`, for example, to scale the peak intensity. This function should take a numeric
        vector as input and return a positive value. If \code{NULL}, then a default function
        will be used. This function is called using the peak intensity as input.}
    \item{plot}{Logical. If \code{FALSE} nothing is plotted. This option may be used to simply
        extract the data from the RI files, but see also \code{\link{ri_data_extract}}.}
    \item{\dots}{Extra arguments, such as \code{pch}, \code{col}, ..., to be passed to the plotting
        functions. See details below.}
}
\details{
    The function uses internally \code{\link{ri_data_extract}} wrapped with a call to
    \code{\link[base]{lapply}}. Because the output of \code{\link[base]{lapply}} is
    a \code{list}, the list is transformed to a matrix by calling \code{rbind}. A column
    of sample indices called \code{sample} is added so each row can be traced back to a sample.

    The \emph{x}-axis of the plot are indexed samples (or RI files), i.e., first sample is 1,
    second 2, and so on. The \emph{y}-axis is retention time (RT) or retention index (RI)
    depending on the value of \code{showRT}. Note that you can search by RT and plot
    by RI depending on the value of \code{useRT}.

    Plot styling can be achieved by passing plotting extra parameters such as \code{col}, \code{pch},
    \code{lty}, and others (see \code{\link[graphics]{par}} and \code{\link[base]{plot}}).
    Because the styles can be applied per sample or per \emph{m/z} value, it is possible
    to prefix a \sQuote{s} to those parameters to specify that the style applies to samples and
    no masses. The available parameters are \code{scol}, \code{spch}, \code{sbg} and \code{scex}.
    See examples and also the notes below.

    The parameter \code{sizefun} is a function that takes a numeric vector as input and
    returns a positive value which correspond to the point size (parameter \code{cex}).
    This function will be applied on the peak intensities and the result will be used as \code{cex}.
    If \code{NULL}, the scaling function is implemented as  \code{(log10(x/100) * (9/8) + (1/2)},
    where \code{x} is the peak intensity. Note that if either \code{cex} or \code{scex} is
    passed to the function, then \code{sizefun} will have no effect.

    The logical parameter \code{plot} controls whether a plot is drawn. This might be useful
    for simply extracting the data for custom plotting with other packages such as \code{ggplot2}.
}
\value{
    Returns \code{\link[base]{invisible}} or a five-column matrix (see below), where the number
    of rows correspond to the number of peaks (or points) in the searched range. If no peaks
    are found in the given range, then it returns \code{NULL}.
    \itemize{
        \item \code{RI} retention time index or the peak.
        \item \code{RT} retention time.
        \item \code{Intensity} peak height intensity.
        \item \code{mz} the \emph{m/z} of the peak.
        \item \code{sample} an integer representing the sample index.
    }
    Compare this list with \code{\link{ri_data_extract}}.
}
\author{Alvaro Cuadros-Inostroza}
\note{
    This function is intended for advanced users and for debugging purposes when \code{\link{TargetSearch}}
    fails to detect a peak.

    Not all graphical parameters passed by \code{\dots} will work, in particular
    \code{panel.first} and \code{panel.last} are known to \emph{not} work.
    Therefore, for more personalized plots, it is recommended to set \code{plot=FALSE}
    and use the returned data for plotting.
}
\seealso{
    \code{\link{plotPeakRI}},
    \code{\link{ri_data_extract}}
}
\examples{

require(TargetSearchData)

# get RI files from TargetSearchData
path <- tsd_data_path()
rifiles <- tsd_rifiles()

# simple plot no style
z <- ri_plot_peak(rifiles, massValues=c(144, 145, 100, 218), timeRange=c(255, 265))
# watch output data
head(z)

# add some style
ri_plot_peak(rifiles, massValues=c(144, 145, 100, 218), timeRange=c(255, 265),
             pch=1:4, col=1:4)
# display the Retention Index instead
ri_plot_peak(rifiles, massValues=c(144, 145, 100, 218), timeRange=c(255, 265),
             pch=1:4, col=1:4, showRT=FALSE)
# use RI index for the time range
ri_plot_peak(rifiles, massValues=c(144, 145, 100, 218), timeRange=c(267000, 275000),
             useRT=FALSE, pch=1:4, col=1:4)
# use styling per sample
scol <- rep(1:5, each=3)
ri_plot_peak(rifiles, massValues=c(144), timeRange=c(255, 265), pch=19, scol=scol)

# using a tsSample object can achieve the same results (RI files are in text format)
smp <- ImportSamplesFromDir(path, ftype = "text")
ri_plot_peak(smp, massValues=c(144), timeRange=c(255, 265), pch=19, scol=scol)

}
% vim: set ts=4 sw=4 et:
