% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/enrichmap.R
\name{enrichmap}
\alias{enrichmap}
\title{Set Enrichment Bubble Heatmap}
\usage{
enrichmap(
  x,
  n_top = 15L,
  set_column = "GeneSet",
  statistic_column = "ZScore",
  contrast_column = "Contrast",
  padj_column = "FDR",
  padj_legend_title = padj_column,
  padj_aggregate_fun = function(padj) median(-log10(padj), na.rm = TRUE),
  padj_cutoff = 0.05,
  plot_sig_only = TRUE,
  padj_fill = "grey",
  colors = c("#3366ff", "darkred"),
  heatmap_color_fun = cameraColorFun,
  scale_by = c("row", "column", "max"),
  cell_size = unit(14, "points"),
  filename,
  height = 5,
  width = 5,
  units = "in",
  heatmap_args = list(),
  padj_args = list(),
  save_args = list(),
  draw_args = list()
)
}
\arguments{
\item{x}{an object that can be coerced to a \code{data.table} with columns
\code{contrast_column}, \code{set_column}, \code{statistic_column}, and
\code{padj_column}.}

\item{n_top}{integer; number of sets (rows) to display. Defaults to the top
15 sets with the highest median \eqn{-log_{10}}(adjusted p-values) across
contrasts.}

\item{set_column}{character; the name of a column in \code{x} containing
unique set identifiers that will be used as the row names in the heatmap.
Default is "GeneSet".}

\item{statistic_column}{character; the name of a column in \code{x}
containing the statistic for each combination of contrast and molecular
signature. Determines the heatmap body colors. Default is "ZScore".}

\item{contrast_column}{character; the name of a column in \code{x} containing
contrasts that will be used as columns for the heatmap. Entries of
\code{x[[rownames_colum]]} must be uniquely defined for each contrast
group.}

\item{padj_column}{character; the name of a column in \code{x} containing the
adjusted p-values. Determines the diameter of each bubble in the heatmap.}

\item{padj_legend_title}{character; title of the background fill legend.
Defaults to \code{padj_column}.}

\item{padj_aggregate_fun}{function; a function used to aggregate the adjusted
p-values in \code{x[[pvalue_column]]} across contrasts for each unique
entry in \code{x[[set_column]]}. The default computes the median of the
\eqn{-log_{10}} adjusted p-values.}

\item{padj_cutoff}{numeric; cutoff for terms to be statistically significant.
If \code{plot_sig_only=TRUE}, only those molecular signatures with at least
one \code{padj_column} value less than this threshold may appear in the
heatmap. Default is 0.05.}

\item{plot_sig_only}{logical; whether to plot only those \code{n_top} terms
that have at least one \code{padj_column} value less than
\code{padj_cutoff}.}

\item{padj_fill}{character; the background color used for values in
\code{padj_column} that are less than \code{padj_cutoff}. Default is
"grey".}

\item{colors}{character; vector of length 2 specifying the colors for the
largest negative and largest positive values of
\code{x[[statistic_column]]}, respectively. Default is "#3366ff" (blue) and
"darkred".}

\item{heatmap_color_fun}{function; used to create the legend for the heatmap
bubble fill. See \code{\link{enrichmapColorFunctions}} for details.}

\item{scale_by}{character; whether to scale the bubbles such that the term
with the largest \eqn{-log_{10}} adjusted p-value in each row
(\code{scale_by="row"}), column (\code{scale_by="column"}), or overall
(\code{scale_by="max"}) is of maximum diameter. Default is "row" to better
visualize patterns across contrasts. May be abbreviated.}

\item{cell_size}{\code{\link[grid]{unit}} object; the size of each heatmap
cell (used for both height and width). Default is \code{unit(14,
  "points")}. This also controls the default text size, which defaults to 90\%
the size of \code{cell_size}.}

\item{filename}{character; the file name used to save the heatmap. If missing
(default), the heatmap will be displayed instead.}

\item{height}{numeric; height of the file in \code{units}.}

\item{width}{numeric; width of the file in \code{units}.}

\item{units}{character; units that define \code{height} and \code{width}.
Defaults to "in" (inches). See \code{\link[grid]{unit}} for possible units.}

\item{heatmap_args}{list; additional arguments passed to
\code{\link[ComplexHeatmap]{Heatmap}}.}

\item{padj_args}{list; additional arguments passed to
\code{\link[ComplexHeatmap]{Legend}}. Modifies the adjusted p-value legend.}

\item{save_args}{list; additional arguments passed to the graphics device
determined by the \code{filename} extension. See
\code{\link[grDevices]{png}} and \code{\link[grDevices]{pdf}} for options.}

\item{draw_args}{list; additional arguments passed to
\code{\link[ComplexHeatmap]{draw-HeatmapList-method}}.}
}
\value{
Nothing. Displays heatmap or saves the heatmap to a file (if
\code{filename} is provided).
}
\description{
Create a bubble heatmap summarizing molecular signature analysis
results, such as those from \code{\link{cameraPR.matrix}}. May also be used
to generate bubble heatmaps of differential analysis results.
}
\details{
The diameter of each bubble is determined by the \eqn{-log_{10}}
adjusted p-values. By default, the bubbles are scaled such that the
contrast with the largest \eqn{-log_{10}} adjusted p-value per row
(\code{scale_by="row"}) has a bubble diameter of \code{0.95 * cell_size},
and all other bubbles in that row are scaled relative to this maximum
diameter; this is to better visualize patterns across contrasts. Bubbles
can also be scaled so that largest \eqn{-log_{10}} adjusted p-value by
column (\code{scale_by="column"}) or in the entire heatmap
(\code{scale_by="max"}) has the maximum diameter. The bubble diameters will
be no smaller than \code{0.2 * cell_size}.
}
\examples{
## Simulate results of cameraPR.matrix
set.seed(1)
df <- 5000L
x <- data.frame(
    Contrast = rep(paste("Contrast", 1:3), each = 4),
    GeneSet = rep(paste("GeneSet", 1:4), times = 3),
    TwoSampleT = 5 * rt(n = 12L, df = df)
)

# Calculate z-statistics, two-sided p-values, and BH adjusted p-values
x$ZScore <- limma::zscoreT(x = x$TwoSampleT, df = df)
x$PValue <- 2 * pnorm(abs(x$ZScore), lower.tail = FALSE)
x$FDR <- p.adjust(x$PValue, method = "BH")

## Plot results
# Same as enrichmap(x, statistic_column = "ZScore")
enrichmap(x = x,
          set_column = "GeneSet",
          statistic_column = "ZScore",
          contrast_column = "Contrast",
          padj_column = "FDR",
          padj_cutoff = 0.05)

# Include gene sets with adjusted p-values above padj_cutoff (0.05).
# Also update adjusted p-value legend title.
enrichmap(x = x,
          statistic_column = "ZScore",
          plot_sig_only = FALSE,
          padj_legend_title = "BH Adjusted\nP-Value")
}
\seealso{
\code{\link[ComplexHeatmap]{ComplexHeatmap-package}}
}
