% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combineRecomputedResults.R
\name{combineRecomputedResults}
\alias{combineRecomputedResults}
\alias{combineCommonResults}
\title{Combine SingleR results with recomputation}
\usage{
combineRecomputedResults(
  results,
  test,
  trained,
  quantile = 0.8,
  fine.tune = TRUE,
  tune.thresh = 0.05,
  assay.type.test = "logcounts",
  check.missing = FALSE,
  warn.lost = TRUE,
  allow.lost = FALSE,
  num.threads = bpnworkers(BPPARAM),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{results}{A list of \link[S4Vectors]{DataFrame} prediction results as returned by \code{\link{classifySingleR}} when run on each reference separately.
This should have the same names as \code{trained}.}

\item{test}{A numeric matrix of single-cell expression values where rows are genes and columns are cells.
Alternatively, a \link[SummarizedExperiment]{SummarizedExperiment} object containing such a matrix.}

\item{trained}{A list of \link[S4Vectors]{List}s containing the trained outputs of multiple references,
equivalent to either (i) the output of \code{\link{trainSingleR}} on multiple references with \code{recompute=TRUE},
or (ii) running \code{trainSingleR} on each reference separately and manually making a list of the trained outputs.
The list should have the same names as \code{results}.}

\item{quantile}{Numeric scalar specifying the quantile of the correlation distribution to use for computing the score, see \code{\link{classifySingleR}}.}

\item{fine.tune}{A logical scalar indicating whether fine-tuning should be performed.}

\item{tune.thresh}{A numeric scalar specifying the maximum difference from the maximum correlation to use in fine-tuning.}

\item{assay.type.test}{An integer scalar or string specifying the assay of \code{test} containing the relevant expression matrix,
if \code{test} is a \link[SummarizedExperiment]{SummarizedExperiment} object.}

\item{check.missing}{Deprecated and ignored, as any row filtering will cause mismatches with the \code{test.genes=} used in \code{\link{trainSingleR}}.}

\item{warn.lost}{Logical scalar indicating whether to emit a warning if markers from one reference in \code{trained} are absent in other references.}

\item{allow.lost}{Deprecated.}

\item{num.threads}{Integer scalar specifying the number of threads to use for index building and classification.}

\item{BPPARAM}{A \link[BiocParallel]{BiocParallelParam} object specifying how parallelization should be performed in other steps,
see \code{?\link{trainSingleR}} and \code{?\link{classifySingleR}} for more details.}
}
\value{
A \link[S4Vectors]{DataFrame} is returned containing the annotation statistics for each cell or cluster (row).
This mimics the output of \code{\link{classifySingleR}} and contains the following fields:
\itemize{
\item \code{scores}, a DataFrame of DataFrames containing the \emph{recomputed} scores for the best label in each reference.
Each nested DataFrame corresponds to a reference and contains \code{labels} (the best label for that cell in this reference) and \code{scores} (the recomputed score).
\item \code{labels}, a character vector containing the per-cell combined label across references.
\item \code{reference}, an integer vector specifying the reference from which the combined label was derived.
\item \code{delta.next}, a numeric vector containing the difference between the best and next-best score.
If \code{fine.tune=TRUE}, this is reported for scores after fine-tuning.
\item \code{orig.results}, a DataFrame containing \code{results}.
}
It may also contain \code{pruned.labels} if these were also present in \code{results}.
}
\description{
Combine results from multiple runs of \code{\link{classifySingleR}} (usually against different references) into a single \link[S4Vectors]{DataFrame}.
This involves recomputing the scores so that they are comparable across references.
}
\details{
Here, the strategy is to perform classification separately within each reference, 
then collate the results to choose the label with the highest score across references.
For a given cell in \code{test}, we extract its assigned label from each reference in \code{results}, along with the marker genes associated with that label.
We take the union of the markers for the assigned labels across all references.
This defines a common feature space in which the score for each reference's assigned label is recomputed using \code{ref};
the label from the reference with the top recomputed score is then reported as the combined annotation for that cell.

A key aspect of this approach is that each entry of \code{results} is generated separately for each reference.
This avoids problems with unintersting technical or biological differences between references that could otherwise introduce noise by forcing irrelevant genes into the marker list. 
Similarly, the common feature space for each cell is defined from the most relevant markers across all references,
analogous to one iteration of fine-tuning using only the best labels from each reference.
Indeed, if fine-tuning is enabled, the common feature space is iteratively refined from the labels with the highest scores, using the same process described in \code{\link{classifySingleR}}.
This allows us to distinguish between closely-related labels from different references.
}
\section{Dealing with mismatching gene availabilities}{

It is recommended that the universe of genes be the same across all references in \code{trained}.
(Or, at the very least, markers used in one reference are available in the others.)
This ensures that a common feature space can be generated when comparing correlations across references.
Differences in the availability of markers between references will have unpredictable effects on the comparability of correlation scores,
so a warning will be emitted by default when \code{warn.lost=TRUE}.
Callers can protect against this by subsetting each reference to the intersection of features present across all references - this is done by default in \code{\link{SingleR}}.

That said, this requirement may be too strict when dealing with many references with diverse feature annotations. 
In such cases, the recomputation for each reference will automatically use all available markers in that reference.
The idea here is to avoid penalizing all references by removing an informative marker when it is only absent in a single reference.
We hope that the recomputed scores are still roughly comparable if the number of lost markers is relatively low,
coupled with the use of ranks in the calculation of the Spearman-based scores to reduce the influence of individual markers.
This is perhaps as reliable as one might imagine.
}

\examples{
# Making up data.
ref <- .mockRefData(nreps=8)
partition <- seq_len(ncol(ref)) \%\% 2
ref1 <- ref[,partition == 0]
ref2 <- ref[,partition == 1]
ref2$label <- tolower(ref2$label) # make lower-case for some more variety.

test <- .mockTestData(ref)

# Performing classification within each reference.
test <- scrapper::normalizeRnaCounts.se(test)

ref1 <- scrapper::normalizeRnaCounts.se(ref1)
train1 <- trainSingleR(ref1, labels=ref1$label)
pred1 <- classifySingleR(test, train1)

ref2 <- scrapper::normalizeRnaCounts.se(ref2)
train2 <- trainSingleR(ref2, labels=ref2$label)
pred2 <- classifySingleR(test, train2)

# Combining results with recomputation of scores.
combined <- combineRecomputedResults(
    results=list(pred1, pred2), 
    test=test,
    trained=list(train1, train2))

combined[,1:5]

}
\references{
Lun A, Bunis D, Andrews J (2020).
Thoughts on a more scalable algorithm for multiple references.
\url{https://github.com/SingleR-inc/SingleR/issues/94}
}
\seealso{
\code{\link{SingleR}} and \code{\link{classifySingleR}}, for generating predictions to use in \code{results}.
}
\author{
Aaron Lun
}
