% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MsBackendMsp.R
\name{MsBackendMsp}
\alias{MsBackendMsp}
\alias{MsBackendMsp-class}
\alias{backendInitialize,MsBackendMsp-method}
\alias{spectraVariableMapping,MsBackendMsp-method}
\alias{export,MsBackendMsp-method}
\title{MS data backend for msp files}
\usage{
\S4method{backendInitialize}{MsBackendMsp}(
  object,
  file,
  mapping = spectraVariableMapping(object),
  ...,
  BPPARAM = SerialParam()
)

MsBackendMsp()

\S4method{spectraVariableMapping}{MsBackendMsp}(object, format = c("msp", "mona"))

\S4method{export}{MsBackendMsp}(
  object,
  x,
  file = tempfile(),
  mapping = spectraVariableMapping(MsBackendMsp()),
  allVariables = TRUE,
  exportName = TRUE,
  ...
)
}
\arguments{
\item{object}{Instance of \code{MsBackendMsp} class.}

\item{file}{\code{character} with the (full) file name(s) of the msp file(s)
from which MS/MS data should be imported or exported.}

\item{mapping}{named \code{character} vector to rename MSP fields to spectra
variables. This allows to correctly import also custom fields or data
from files with different MSP \emph{flavors}.}

\item{...}{Currently ignored.}

\item{BPPARAM}{Parameter object defining the parallel processing
setup to import data in parallel. Defaults to \code{BPPARAM = SerialParam()}. See \code{\link[BiocParallel:register]{BiocParallel::bpparam()}} for more information.
Parallel processing would make most sense for import from a large
set of individual MSP files, but could also improve performance for
import from a (very large) single MSP file.}

\item{format}{For \code{spectraVariableMapping()}: \code{character(1)} specifying for
which MSP \emph{flavour} the mapping should be returned. Currently supported
are: \code{format = "msp"} (generic MSP format, for example for MS-DIAL MSP
files) and \code{format = "mona"} (MSP files in MoNA flavour).}

\item{x}{For \code{export()}: a \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object that should be
exported to the specified MSP file.}

\item{allVariables}{\code{logical(1)} whether all spectra variables in \code{x}
should be exported or only those defined with \code{mapping}.}

\item{exportName}{\code{logical(1)} whether a \code{NAME} field should always be
exported even if not provided in \code{x}.}
}
\value{
\code{MsBackendMsp()} and \code{backendInitialize()} return an instance of a
\code{MsBackendMsp} class. \code{spectraVariableMapping()} a named \code{character}
vector with the mapping between spectra variables and MSP data fields.
}
\description{
The \code{MsBackendMsp} class supports import of MS/MS spectra data from
files in NIST MSP file format. \code{MsBackendMsp} extends the
\code{\link[Spectra:MsBackend]{Spectra::MsBackendDataFrame()}} backend directly and supports thus the
\code{\link[Spectra:addProcessing]{Spectra::applyProcessing()}} function to make data manipulations persistent.

New objects are created with the \code{MsBackendMsp()} function. The
\code{backendInitialize()} method has to be subsequently called to
initialize the object and import MS/MS data from (one or more) msp
files.

The \code{MsBackendMsp} backend provides an \code{export()} method that allows to
export the data from the \code{Spectra} object (parameter \code{x}) to a file in
MSP format.

Parameters to this function are:
\itemize{
\item \code{x}: the \code{Spectra} object that should be exported.
\item \code{file}: \code{character(1)} with the desired file name.
\item \code{mapping}: named \code{character} providing the mapping between spectra
variables and MSP data fields. Defaults to
\code{mapping = spectraVariableMapping(MsBackendMsp())}.
\item \code{allVariables}: \code{logical(1)} whether all spectra variables in \code{x} should be
exported or only those defined with \code{mapping}.
\item \code{exportName}: \code{logical(1)} whether a \code{NAME} field should always be exported
even if not provided in \code{x}.
}

See the package vignette for details and examples.

The \code{spectraVariableMapping()} function allows to provide the mapping
between spectra variable names (i.e. the names that will be used for the
spectra variables in the \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object) and the data field
names of the MSP file. Parameter \code{format} allows to select pre-defined
mappings. Currently supported mapping flavors are:
\itemize{
\item \code{format = "msp"}: default MSP field names. Should work with standard NIST
MSP files or MSP files exported from MS-DIAL.
\item \code{format = "mona"}: MSP file format from MoNA including LipidBlast.
}
}
\note{
Format requirements/assumptions of MSP files:
\itemize{
\item Comment lines are expected to start with a \verb{#}.
\item Multiple spectra within the same MSP file are separated by an empty line.
\item The first n lines of a spectrum entry represent metadata.
\item Metadata is provided as "name: value" pairs (i.e. name and value separated
by a ":").
\item One line per mass peak, with values separated by a whitespace or tabulator.
\item Each line is expected to contain at least the m/z and intensity values (in
that order) of a peak. Additional values are currently ignored.
}
}
\examples{

## Import spectra from a MSP file from LipidBlast
f <- system.file("extdata", "small-export-LipidBlast.msp",
    package = "MsBackendMsp")
be <- backendInitialize(MsBackendMsp(), f)
be

be$msLevel
be$intensity
be$mz

## precursor m/z are however all missing
be$precursorMz

## Default spectra variable mapping
spectraVariableMapping(MsBackendMsp())

## In fact, to read MSP files in "LipidBlast flavour" (same as MoNA) we
## should use a different spectra variable mapping
spectraVariableMapping(MsBackendMsp(), "mona")

## Importing the data with this will correctly retrieve data
be <- backendInitialize(MsBackendMsp(), f,
    mapping = spectraVariableMapping(MsBackendMsp(), "mona"))
be$precursorMz

## Other fields are also correctly mapped, but might need to be converted
## to e.g. numeric, such as "exactmass"
be$exactmass

be$exactmass <- as.numeric(be$exactmass)

be$adduct
be$formula

## Exporting Spectra objects in MSP format.

sps <- Spectra(be)
export(MsBackendMsp(), sps, file = stdout())
}
\author{
Steffen Neumann, Michael Witting, Laurent Gatto and Johannes Rainer
}
