% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/enhance_table.R
\name{enhance_table}
\alias{enhance_table}
\title{Visually enhances a functional enrichment result table}
\usage{
enhance_table(
  res_enrich,
  res_de,
  annotation_obj,
  gtl = NULL,
  n_gs = 50,
  gs_ids = NULL,
  chars_limit = 70,
  plot_style = c("point", "ridgeline"),
  ridge_color = c("gs_id", "gs_score"),
  plot_title = NULL
)
}
\arguments{
\item{res_enrich}{A \code{data.frame} object, storing the result of the functional
enrichment analysis. See more in the main function, \code{\link[=GeneTonic]{GeneTonic()}}, to check the
formatting requirements (a minimal set of columns should be present).}

\item{res_de}{A \code{DESeqResults} object.}

\item{annotation_obj}{A \code{data.frame} object with the feature annotation.
information, with at least two columns, \code{gene_id} and \code{gene_name}.}

\item{gtl}{A \code{GeneTonic}-list object, containing in its slots the arguments
specified above: \code{dds}, \code{res_de}, \code{res_enrich}, and \code{annotation_obj} - the names
of the list \emph{must} be specified following the content they are expecting}

\item{n_gs}{Integer value, corresponding to the maximal number of gene sets to
be displayed.}

\item{gs_ids}{Character vector, containing a subset of \code{gs_id} as they are
available in \code{res_enrich}. Lists the gene sets to be displayed.}

\item{chars_limit}{Integer, number of characters to be displayed for each
geneset name.}

\item{plot_style}{Character value, one of "point" or "ridgeline". Defines the
style of the plot to summarize visually the table.}

\item{ridge_color}{Character value, one of "gs_id" or "gs_score", controls the
fill color of the ridge lines. If selecting "gs_score", the \code{z_score} column
must be present in the enrichment results table - see \code{get_aggrscores()} to do
that.}

\item{plot_title}{Character string, used as title for the plot. If left \code{NULL},
it defaults to a general description of the plot and of the DE contrast.}
}
\value{
A \code{ggplot} object
}
\description{
Creates a visual summary for the results of a functional enrichment analysis,
by displaying also the components of each gene set and their expression change
in the contrast of interest
}
\examples{

library("macrophage")
library("DESeq2")
library("org.Hs.eg.db")
library("AnnotationDbi")

# dds object
data("gse", package = "macrophage")
dds_macrophage <- DESeqDataSet(gse, design = ~ line + condition)
rownames(dds_macrophage) <- substr(rownames(dds_macrophage), 1, 15)
dds_macrophage <- estimateSizeFactors(dds_macrophage)

# annotation object
anno_df <- data.frame(
  gene_id = rownames(dds_macrophage),
  gene_name = mapIds(org.Hs.eg.db,
    keys = rownames(dds_macrophage),
    column = "SYMBOL",
    keytype = "ENSEMBL"
  ),
  stringsAsFactors = FALSE,
  row.names = rownames(dds_macrophage)
)

# res object
data(res_de_macrophage, package = "GeneTonic")
res_de <- res_macrophage_IFNg_vs_naive

# res_enrich object
data(res_enrich_macrophage, package = "GeneTonic")
res_enrich <- shake_topGOtableResult(topgoDE_macrophage_IFNg_vs_naive)
res_enrich <- get_aggrscores(res_enrich, res_de, anno_df)
enhance_table(res_enrich,
  res_de,
  anno_df,
  n_gs = 10
)

# using the ridge line as a style, also coloring by the Z score
res_enrich_withscores <- get_aggrscores(
  res_enrich,
  res_de,
  anno_df
)
enhance_table(res_enrich_withscores,
  res_de,
  anno_df,
  n_gs = 10, 
  plot_style = "ridgeline",
  ridge_color = "gs_score"
)
}
