% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/degCreCore.R
\name{optimizeAlphaDegCre}
\alias{optimizeAlphaDegCre}
\title{Run DegCre with DEG alpha optimization.}
\usage{
optimizeAlphaDegCre(
  DegGR,
  DegP,
  DegLfc = NULL,
  CreGR,
  CreP,
  CreLfc = NULL,
  reqEffectDirConcord = TRUE,
  padjMethod = "bonferroni",
  maxDist = 1e+06,
  verbose = FALSE,
  smallestTestBinSize = 100,
  fracMinKsMedianThresh = 0.2,
  testedAlphaVals = c(0.005, 0.01, 0.02, 0.03, 0.05, 0.1),
  minNDegs = 5
)
}
\arguments{
\item{DegGR}{A \link[GenomicRanges]{GRanges} object of gene TSSs. Multiple
TSSs per gene are allowed.}

\item{DegP}{A numeric vector of differential expression p-values for genes
in \code{DegGR}.}

\item{DegLfc}{A numeric vector of log fold-change values of differential
expression for gene in \code{DegGR}. Required when
\code{reqEffectDirConcord = TRUE}. (Default: \code{NULL})}

\item{CreGR}{A \link[GenomicRanges]{GRanges} object of CRE regions.}

\item{CreP}{A numeric vector differential signal p-values for regions in
\code{CreGR}.}

\item{CreLfc}{A numeric vector log fold-change values of differential
signal for regions in \code{CreGR}. Required when
\code{reqEffectDirConcord = TRUE}. (Default: \code{NULL})}

\item{reqEffectDirConcord}{A logical whether to require concordance between
the effect direction between DEG and CRE differential values.
(Default: \code{NULL})}

\item{padjMethod}{A character value indicating the method for p-value
adjustment. Do not change from default under most circumstances. Can be any
method name accepted by \code{p.adjust()} (Default: \code{bonferroni})}

\item{maxDist}{An integer value specifying the maximum distance for
probability calculation of TSS to CRE associations. (Default: \code{1e6})}

\item{verbose}{A logical indicating whether to print messages of step
completion and algorithm results. (Default: \code{NULL})}

\item{smallestTestBinSize}{An integer value specifying the size
(number of elements) of the smallest distance bin to be considered in the
optimization algorithm. (Default: \code{100})}

\item{fracMinKsMedianThresh}{A numeric value between 0 and 1 specifying the
optimization criterion for the distance bin size algorithm (See Details).
(Default: \code{0.2})}

\item{testedAlphaVals}{A numeric vector of DEG alpha values to test
(Default: \code{c(0.005,0.01,0.02,0.03,0.05,0.1)}).}

\item{minNDegs}{An integer specifying minimum number of DEGs that pass
the lowest \code{testedAlphaVals}. (Default: \code{5})}
}
\value{
A named list containing:
\describe{
  \item{alphaPRMat}{A matrix of Precision-Recall Area Under the Curve
  (AUC) values.}
  \item{degCreResListsByAlpha}{Named list of DegCre results lists indexed
  by the \code{testedAlphaVals}}.
}

The columns of \code{alphaPRMat} are:
\describe{
  \item{alphaVal}{Numeric vector of tested DEG alpha value.}
  \item{AUC}{Numeric vector of Area under the curve of a Precision-Recall
  (PR) curve based on associations recovering significant DEGs.}
  \item{deltaAUC}{Numeric vector of PR AUC minus the AUC of the
  no-skill line.}
  \item{normDeltaAUC}{Numeric vector of the value of \code{deltaAUC}
  divided by one minus the no-skill AUC.}
}
}
\description{
Runs DegCre across a set of DEG alpha thesholds to find optimal performance.
}
\details{
This function runs \link{runDegCre} for each value in \code{testedAlphaVals}.
The performance at each tested alpha is evaluated with \link{degCrePRAUC}.
which generates a Precision-Recall curve based on the recovery rate of DEGs
by associations.
Various AUCs are calculated as performance metrics. Using the alpha with
the highest value of
\code{normDeltaAUC} is recommended (see Examples).
}
\examples{
#Load required packages.
library(GenomicRanges)

#Load sample data.
data(DexNR3C1)

subDegGR <-
 DexNR3C1$DegGR[which(Seqinfo::seqnames(DexNR3C1$DegGR)=="chr1")]
subCreGR <-
 DexNR3C1$CreGR[which(Seqinfo::seqnames(DexNR3C1$CreGR)=="chr1")]

# Run DegCre over range of alpha values:
alphaOptList <- optimizeAlphaDegCre(DegGR = subDegGR,
                           DegP = subDegGR$pVal,
                           DegLfc = subDegGR$logFC,
                           CreGR = subCreGR,
                           CreP = subCreGR$pVal,
                           CreLfc = subCreGR$logFC)

bestAlphaId <- which.max(alphaOptList$alphaPRMat[,4])
bestDegCreResList <- alphaOptList$degCreResListsByAlpha[[bestAlphaId]]

}
\author{
Brian S. Roberts
}
